/* 
 *  grhtml.c: Formatting for "grind"-ing program listings.  This
 *      is intended to be a layer between the (language-specific, 
 *      formatter-independent) pattern recognition layer and the 
 *      (language-independent, formatter-specific) document driver
 *      layer of a listing-grinding application. 
 *      Uses the same interface as grmif (the FrameMaker conversion).
 */


#include "gr.h"    /* header file for this package */
#include "htmldrvr.h"  /* driver for building Maker Interchange Format */
#include "groptions.h"
#include <stdio.h>   /* For error messages ONLY */

/* ----------------------------------------------------------
 * Mapping from grind categories to HTML highlight tags is here in one
 *   place for maintainability.  Each category is associated 
 *   either with a tag name (e.g., "BF" for <BF>string</BF>)
 *   or with a NULL pointer in case no tag should be used.
 *
 * Note: This strategy will change completely when HTML version 
 *    3 brings style sheets.  Then this mapping will be moved to
 *    style sheets, and the symbolic naming strategy as in 
 *    the FrameMaker/MIF version will be used. 
 *------------------------------------------------------------
 */

typedef char * StyleName; 
#define NONE (char *) 0

/* This first set is LOGICAL, but since current browsers are
 * quite illogical (e.g., NetScape doesn't honor DFN at all!!)
/* static StyleName Keyword_Style  = "STRONG";   /* Strong (bold) */
/* static StyleName Identifier_Style = NONE;     
/* static StyleName Declaration_Style = "DFN";   /* Definition (bold italic)*/
/* static StyleName Comment_Style = "EM";        /* Emphasized (italics) */
/* static StyleName Quoted_Style = "KBD";        /* Keyboard (typeweriter) */
/* */

/* This second set resorts to PHYSICAL formatting for some tags 
 * and is tuned for NetScape (UGH, I didn't wanna do it this way)
 */
static StyleName Keyword_Style  = "B";   /* Strong (bold) */
static StyleName Identifier_Style = NONE;     
static StyleName Declaration_Style = "STRONG";   /* Definition (bold italic)*/
static StyleName Comment_Style = "EM";        /* Emphasized (italics) */
static StyleName Quoted_Style = "KBD";        /* Keyboard (typeweriter) */

/* ----------------------------------------------------------
 *  END of map to highlight styles
 * 
 *------------------------------------------------------------
 */

int line_num = 1;  /* Bumped by newline() */

/* line_num_output:
 *  Call for EVERY line if line numbering is enabled *
 */
void line_num_output(int n); 

void init_gr() { 
    init_html(); 
    line_num = 1; 
    if (optblock.line_nums > 0) {
	line_num_output(line_num); 
    }
}

void end_gr()  { end_html();  }


void gr_error(char *context, char *badstring) {
  fprintf(stderr, "*Grind Error: %s\t%s\n", context, badstring); 
}

/*  gr_trace is for debugging only, and is not enabled while
 * creating html code 
 */
void gr_trace(char *whatpattern) {
  return;
}

void
keyword(char *s) {
  html_string_in_style(s, Keyword_Style); 
}

void
identifier(char *s) {
  html_string_in_style(s, Identifier_Style); 
}

void
declaration(char *s) {
  html_begin_larger(1);
  html_string_in_style(s, Declaration_Style); 
  html_end_larger();
}

void blanks(char *s) {
  html_string(s); 
}

void comment(char *s) {
  html_string_in_style(s, Comment_Style); 
}

void 
quoted(char *s) {
  html_string_in_style(s, Quoted_Style); 
}



void
newline() {
    html_newline(); 

    ++ line_num; 

    /* Optionally prints line number.  In HTML 3.0 these should
     * go at end of line (using TAB feature).  Since most browsers
     * don't support that yet, I'll put them at the beginning of the 
     * line.  This has a BUG: POSSIBLE EXTRA LINE AT END OF FILE
     * but I'll let that go for now because it will be fixed when
     * HTML 3.0 features are widely available, which should be soon. 
     */
    if (optblock.line_nums > 0) {
	line_num_output(line_num); 
    }
}

/* LINE NUMBERING */
/* To keep things as even as possible, we'll do everything the
 * same *except* that we print blanks if there is no line# to
 * be printed here.  Note this doesn't quite work with 
 * proportional fonts --- which is why we *really* need HTML3 tabs
 */
void 
line_num_output(int n){
	char buf[15]; 	
	html_begin_larger(-3); 
	if ((line_num % optblock.line_nums) ==  0) {
	    sprintf( buf, "%4d  ", line_num);
		html_string_in_style( buf, "EM"); 
	} else {
	    html_string_in_style( "      ", "EM"); 
	}
	html_end_larger(); 
}


void 
echo_char(char ch) {
  html_echo_char(ch); 
}

#define MAX_MARKERS 2000
static char *marker_list[MAX_MARKERS];
static int next_marker=0;

/* 
 * Leave cross-reference markers for declarations - 
 *    Argument is the string to be put in the index
 *    Note: We should also build an index document here.
 * 
 * Note:  As of 19 June, this is a deprecated function
 * 
 *
 */
 void mark_declaration(char *index_string) {
  char marker_string[20];
  sprintf(marker_string, "M_%03d", next_marker); 
  html_marker_string(marker_string, index_string); 
  if (next_marker < MAX_MARKERS) {
    marker_list[next_marker] = (char *) strdup(index_string);
    ++ next_marker;
  }
}

 void begin_mark_declaration(char *index_string) {
  char marker_string[20];
  sprintf(marker_string, "M_%03d", next_marker); 
  html_open_marker(marker_string, index_string); 
  if (next_marker < MAX_MARKERS) {
    marker_list[next_marker] = (char *) strdup(index_string);
    ++ next_marker;
  }
}

 void end_mark_declaration(char *index_string) {
  html_close_marker(); 
}



/*
 * The dumping should be designed to facilitate matching
 * with awk or perl scripts, to provide an easy way to 
 * customize how indexes are build and used. 
 */
void dump_markers() {
  int i;
  for (i=1; i<next_marker; ++i) {
    /* Dump in comment in easy-to-match format */
  }
}

/* Source identification functions */


void
title(char *s) {
  html_page_title(s); 
}

void
heading(char *tag, char *content) {
    /* Nothing for now */
}

/* Tagged conditional text */

void begin_block_comment() {
    /* Nothing for now */
}

void end_block_comment() {
    /* Nothing for now */
}

void gr_figure(char *kind, char *filename) {
     html_image_reference(kind, filename); 
   }


/* Symbols - those that are printed as themselves, and 
 *  those that aren't. 
 * For now, we'll just echo all the characters. This will 
 * get much, much better with equation support in HTML 3.
 * 
 */

void echo_symbol(char ch) {
  html_echo_char(ch); 
}

void s_neq(char *s) { html_string(s); }
void s_leq(char *s) { html_string(s); }
void s_geq(char *s) { html_string(s); }
void s_Rtarrow(char *s) { html_string(s); }
void s_not(char *s) { html_string(s); }
void s_bullet(char *s) { html_string(s); }






