/* 
 *  grhtml.c: Formatting for "grind"-ing program listings.  This
 *      is intended to be a layer between the (language-specific, 
 *      formatter-independent) pattern recognition layer and the 
 *      (language-independent, formatter-specific) document driver
 *      layer of a listing-grinding application. 
 *      Uses the same interface as grmif (the FrameMaker conversion).
 */


#include "gr.h"    /* header file for this package */
#include "htmldrvr.h"  /* driver for building Maker Interchange Format */
#include "groptions.h"
#include "html_styles.h"  /* Logical categories to html tags */
#include <stdio.h>   /* For error messages ONLY */

/* ----------------------------------------------------------
 * Mapping from grind categories to HTML highlight tags ---
 *   place for maintainability.  Each category is associated 
 * The rationale for putting the mapping here, rather than in 
 * the driver, is that it is partly application-dependent, e.g., 
 * the particular logical styles supported are specific to 
 * grinding program listings.  (But I'm still not quite comfortable
 * with that decision.) 
 * 
 *
 * Note: This strategy will change completely when HTML CSS1
 *    style sheets are widely supported.  Then this mapping will be moved to
 *    style sheets, and the symbolic naming strategy as in 
 *    the FrameMaker/MIF version will be used. 
 *------------------------------------------------------------
 */

typedef char * StyleName; 

static StyleName Keyword_Style_Begin  = "";  
static StyleName Keyword_Style_End  = "";  
static StyleName Identifier_Style_Begin = "";
static StyleName Identifier_Style_End = "";
static StyleName Declaration_Style_Begin = "";
static StyleName Declaration_Style_End = "";
static StyleName Comment_Style_Begin = "";
static StyleName Comment_Style_End = "";
static StyleName Quoted_Style_Begin = "";
static StyleName Quoted_Style_End = "";


/* ----------------------------------------------------------
 *  END of map to highlight styles
 * 
 *------------------------------------------------------------
 */

int line_num = 1;  /* Bumped by newline() */

/* line_num_output:
 *  Call for EVERY line if line numbering is enabled *
 */
void line_num_output(int n); 

void init_gr() { 
    init_html(); 

    /* Logical to physical style mappings */
    init_html_styles(); 
    html_style_map("Keyword", &Keyword_Style_Begin, &Keyword_Style_End); 
    html_style_map("Declaration", &Declaration_Style_Begin, 
		   &Declaration_Style_End); 
    html_style_map("Comment", &Comment_Style_Begin, &Comment_Style_End); 
    html_style_map("Quoted", &Quoted_Style_Begin, &Quoted_Style_End); 

    line_num = 1; 
    if (optblock.line_nums > 0) {
	line_num_output(line_num); 
    }
}

void end_gr()  { end_html();  }


void gr_error(char *context, char *badstring) {
  fprintf(stderr, "*Grind Error: %s\t%s\n", context, badstring); 
}

/*  gr_trace is for debugging only, and is not enabled while
 * creating html code 
 */
void gr_trace(char *whatpattern) {
  return;
}

void
keyword(char *s) {
  html_styled_string(s, Keyword_Style_Begin, Keyword_Style_End); 
}

void
identifier(char *s) {
  html_styled_string(s, Identifier_Style_Begin, Identifier_Style_End); 
}

void
declaration(char *s) {
  html_styled_string(s, Declaration_Style_Begin, Declaration_Style_End); 
}

void blanks(char *s) {
  html_string(s); 
}

void comment(char *s) {
  html_styled_string(s, Comment_Style_Begin, Comment_Style_End); 
}

void 
quoted(char *s) {
  html_styled_string(s, Quoted_Style_Begin, Quoted_Style_End); 
}


void
newline() {
    html_newline(); 

    ++ line_num; 

    /* Optionally prints line number.  In HTML 3.0 these should
     * go at end of line (using TAB feature).  Since most browsers
     * don't support that yet, I'll put them at the beginning of the 
     * line.  This has a BUG: POSSIBLE EXTRA LINE AT END OF FILE
     * but I'll let that go for now because it will be fixed when
     * HTML 3.0 features are widely available, which should be soon. 
     */
    if (optblock.line_nums > 0) {
	line_num_output(line_num); 
    }
}

/* LINE NUMBERING */
/* To keep things as even as possible, we'll do everything the
 * same *except* that we print blanks if there is no line# to
 * be printed here.  Note this doesn't quite work with 
 * proportional fonts --- which is why we *really* need HTML3 tabs
 */
void 
line_num_output(int n){
	char buf[15]; 	
	html_begin_larger(-3); 
	if ((line_num % optblock.line_nums) ==  0) {
	    sprintf( buf, "%4d  ", line_num);
	    html_styled_string(buf, "<EM>", "</EM>"); 
	} else {
	    html_styled_string( "      ", "<EM>", "</EM>"); 
	}
	html_end_larger(); 
}


void 
echo_char(char ch) {
  html_echo_char(ch); 
}

#define MAX_MARKERS 2000
static char *marker_list[MAX_MARKERS];
static int next_marker=0;

/* 
 * Leave cross-reference markers for declarations - 
 *    Argument is the string to be put in the index
 *    Note: We should also build an index document here.
 * 
 * Note:  As of 19 June, this is a deprecated function
 * 
 *
 */
 void mark_declaration(char *index_string) {
  char marker_string[20];
  sprintf(marker_string, "M_%03d", next_marker); 
  html_marker_string(marker_string, index_string); 
  if (next_marker < MAX_MARKERS) {
    marker_list[next_marker] = (char *) strdup(index_string);
    ++ next_marker;
  }
}

 void begin_mark_declaration(char *index_string) {
  char marker_string[20];
  sprintf(marker_string, "M_%03d", next_marker); 
  html_open_marker(marker_string, index_string); 
  if (next_marker < MAX_MARKERS) {
    marker_list[next_marker] = (char *) strdup(index_string);
    ++ next_marker;
  }
}

 void end_mark_declaration(char *index_string) {
  html_close_marker(); 
}



/*
 * The dumping should be designed to facilitate matching
 * with awk or perl scripts, to provide an easy way to 
 * customize how indexes are build and used. 
 */
void dump_markers() {
  int i;
  for (i=1; i<next_marker; ++i) {
    /* Dump in comment in easy-to-match format */
  }
}

/* Source identification functions */


void
title(char *s) {
  html_page_title(s); 
}

void
heading(char *tag, char *content) {
    /* Nothing for now */
}

/* Tagged conditional text */

void begin_block_comment() {
    /* Nothing for now */
}

void end_block_comment() {
    /* Nothing for now */
}

void gr_figure(char *kind, char *filename) {
     html_image_reference(kind, filename); 
   }


/* Symbols - those that are printed as themselves, and 
 *  those that aren't. 
 * 
 */

void echo_symbol(int glyphcode, char  *str) 
{
  if (optblock.glyph_mapping) 
    {
      html_symbol(glyphcode, str); 
    }
  else
    {
      html_string(str);
    }
}







