
/*
 * HTML driver for source code pretty-printers
 * 
 * 
 * Author: Michal Young Created: 24 Aug 91 for LaTeX Revised 30 Apr 94 for MIF
 * Revised 9 July 94 for ada2mif Separated 11 July 94 as a library Revised 23
 * Febuary 95 for HTML
 * 
 */
static char     VersionID[] = "$Id: htmldrvr.c,v 1.13 1996/11/02 20:55:26 young Exp $";

#include "htmldrvr.h"
#include "groptions.h"
#include <ctype.h>
#include <string.h>

/*************************************************************
 *  Internal utilities (not in .h file)
 *************************************************************/


/* Translation tables for special characters */
void            init_charmap();

/*
 * Purify a string: escape whatever needs escaping Pass it a buffer large
 * enough for the purified string to fit (allow about 3 times max string
 * length) Return values is a shared reference to the buffer, for convenience
 * in using Purify_String in a call to printf. Do NOT use the same buffer in
 * two nested function calls (e.g., a printf with two string arguments),
 * since this would alias the buffer.
 */
char           *Purify_String(char *s, char *buffer);

static char     buf[600];	/* Where only one buffer is needed, this will
				 * do */

/*************************************************************/
/* Initiation and shutdown (required!)                      */
/*************************************************************/
/*
 * Since we don't know whether this will be a complete program listing or
 * just a snippet included in something else, we won't generate the <HEAD>
 * and <BODY> structural framework --- Those can be spliced in under control
 * of a script.
 */

void
init_html()
{
	init_charmap();
	printf("<!-- The following was generated by htmldrvr.c, -->\n"
	       "<!-- Version %s -->\n"
	       "<!-- part of the src2fm package for source code -->\n"
	       "<!-- formatting -->\n",
	       VersionID);
	printf("<PRE>\n");
};

void
end_html()
{
	printf("</PRE>\n");
};


/*************************************************************
 * Heading elements
/*************************************************************/
void
html_page_title(char *s)
{
	printf("<H1>%s</H1>\n", Purify_String(s, buf));
}


/*************************************************************/
/* The following sets substrings within a line               */
/* with an appropriate character tag                       */
/*************************************************************/

void
html_string(char *s)
{
	printf("%s", Purify_String(s, buf));
};

void
html_string_in_style(char *s, char *tag)
{
	Purify_String(s, buf);
	if (tag) {
		printf("<%s>%s</%s>", tag, Purify_String(s, buf), tag);
	} else {
		printf("%s", buf);
	}
}

/* ************** */
/* The above function html_string_in_style is ALMOST good enough, but */
/* when we have complex tags, and possibly groups of tags, we need    */
/* to specify the beginning and ending tag sequences separately.      */
/* One could argue that we should be passed only a single logical tag, */
/* and map it to a sequence of html tags here .... but then we put the */
/* semantics-to-presentation mapping in the driver, making it depend on */
/* a particular application (e.g., source code listings).  So, instead */
/* we keep this function simple and do the tag translation in the */
/* application-dependent part. */

void html_styled_string(char *s, char *begin_tags, char *end_tags)
{
  printf("%s%s%s", begin_tags, Purify_String(s, buf), end_tags);
}



/*
 * An invisible marker in the text, for indexing Second argument is a string
 * name that can be placed in an index. The marker should already by
 * html-pure (i.e., not containing any special characters.)
 * 
 */
void 
html_marker_string(char *marker, char *index)
{
	printf("<A NAME=\"%s\" TITLE=\"%s\"> </A>",
	       marker, Purify_String(index, buf));
}

void html_open_marker(char *marker, char *index) 
{
	printf("<A NAME=\"%s\" TITLE=\"%s\">",
	       marker, Purify_String(index, buf));
}


void html_close_marker() {
	printf("</A>"); 
}

/* 
 * Images 
 * 
 * At this point, the only kind of image conversion I will attempt
 *   is from PostScript into GIF.  A more general, table-driven
 *   scheme may be needed  sometime in the future.
 */


int is_postscript_mime (char *s) 
{
  /* Only application/postscript is correct, but let's be lenient */
  static char *postscript_mime_types[] = 
  { "application/postscript", "application/x-postscript",
    "image/postscript", "image/x-postscript", 0
  };
  int i;

  for (i = 0; postscript_mime_types[i]; ++i)
    { if(strcasecmp(s, postscript_mime_types[i])==0)
      { return 1; }
    }
  return 0;
}
	


void html_image_reference(char *kind, char * filename) {
  int dotpos = 0; /* File extension position in string */
  char namebuf[200]; 
  int i;

  
  printf("<!-- GRAPHIC kind %s -->\n", kind); 
  if (  optblock.image_conversion
	&& is_postscript_mime(kind)
      ) 
    {
      strcpy(namebuf, filename); 
      for (i=0; namebuf[i]; ++i)
	{
	  if (filename[i] == '.') dotpos = i;
	}
      if (dotpos > 0) 
	{
	  strcpy(namebuf+dotpos+1, "gif");
	}
      else 
	{
	  strcat(namebuf, ".gif");
	}
      printf("<!-- CONVERT  %s %s  -->\n", filename, namebuf); 
      printf("<IMG SRC=\"%s\">\n", namebuf); 
    }
  else
    {
      printf("<IMG SRC=\"%s\">\n", filename); 
    }
}

/*
 * Increase/decrease font size
 */
static int      current_size = 1;
void 
html_begin_larger(int delta)
{
	++current_size;
	printf("<Font Size=\"%+d\">", delta);
}

void 
html_end_larger()
{
	--current_size;
	printf("</Font>");
}


/**************************************************************
 * The following set individual characters or special symbols;
 * If a character needs to be escaped, it
 * will be.
 **************************************************************/


/*
 * Escmap[ch] = 0 if ch can be echoed as itself, -1 if ch should be printed
 * as decimal code
 */
static int      escmap[256];

void
init_charmap()
{
	/*
	 * We will escape: Non-graphic characters and the html special
	 * characters < > & "
	 */
	static char    *impure = "<>&\"\\";
	char            ch;
	for (ch = 0; ch <= 126; ++ch) {
		escmap[ch] = 0;	/* Default: just print it */
		if (!isprint(ch)) {
			escmap[ch] = -1;
		};
	};
	{
		int             i;
		for (i = 127; i <= 255; ++i) {
			escmap[(char) i] = -1;
		};
		for (i = 0; ch = impure[i]; ++i) {
			escmap[impure[i]] = -1;
		};
	};			/* Nested declaration block */
}

void
html_echo_char(char ch)
{
	if (escmap[ch] == 0) {
		putchar(ch);
	} else if (escmap[ch] == -1) {
		printf("&#%03d;", ch);
	} else {
		putchar('?');
	};
};

/*
 * Purify a string: escape whatever needs escaping Pass it a buffer large
 * enough for the purified string to fit (max 5 times string length). Return
 * values is a shared reference to the buffer, for convenience in using
 * Purify_String in a call to printf. Do NOT use the same buffer in two
 * nested function calls (e.g., a printf with two string arguments), since
 * this would alias the buffer.
 */
char           *
Purify_String(char *s, char *buffer)
{
	char            ch;
	int             si, bi;	/* Index into input string, output buffer */
	for (si = 0, bi = 0; s[si]; si++) {
		ch = s[si];
		if (escmap[ch] == 0) {
			buffer[bi++] = ch;
		} else if (escmap[ch] == -1) {
			sprintf(&buffer[bi], "&#%03d;", ch);
			bi += 6;
		} else {
			buffer[bi++] = '?';
		};
	};
	buffer[bi] = 0;
	return buffer;
}

void 
html_newline()
{
	putchar('\n');
}
