##
## glyphmaps.awk: 
## 
##    Process a glyph map file, together with the glyph list, and 
##    produced on the output a table that can be used to translate 
##    glyph codes to text sequences, e.g., to TeX commands or to 
##    MIF commands or HTML entities.  
##
##    Glyph list should be read first, followed by glyph map table. 
##    For example, a command line could be 
##        nawk -f glyphmaps.awk glyphcodes.src glyphs_tex.src >glyphs_tex.h
##
##    We read the glyph list for two reasons: 
##       1: to determine the internal code number used for each glyph.
##          This is determined by the order of glyphs in the list, and
##          may differ from the order of glyphs in the map file for any
##          particular back end. 
##       2: to spot any missing or undefined glyphs, which are reported.
##          This helps us to maintain consistency among all the different
##          back ends and front ends. 
##
##
BEGIN { 
  glyph_count = 0;  ## Used to generate serial numbers of glyphs
  phase = "NONE"; 
  errout = "/dev/stderr"; 

  print "/* DO NOT EDIT THIS FILE */";
  print "/* It was generated automatically by glyphmaps.awk */";
  print "/* Edit the glyph list (e.g. glyphcodes.src) and the */"
  print "/* language-specific mapping table (e.g. glyphs_tex.src) */"
  print "/* and then regenerate this file using make. */"
  print "/* Generated on ";
  system("date");
  print "*/"

}

/^%/ { ## Literal inclusion 
  print substr($0,2,length($0)-1);
}

/^BEGIN LIST/ { phase = "LIST"; next; }
/^END LIST/ { phase = "NONE"; next;  }

/^BEGIN MAP/ { phase = "MAP"; next; }
/^END MAP/ { phase = "DONE"; next; }

/^[a-zA-Z]/ {
  if (phase == "LIST") 
    {
      glyph_list_line(); 
    }
  else if (phase == "MAP") 
    {
      glyph_map_line();
    }
  else if (phase == "DONE")
    {
      print "glyphmaps.awk confused in file " FILENAME  >errout;
      print "   Line " FNR " occurs after end of glyph map file " >errout;
    }
  else
    {
      print "glyphmaps.awk confused at line " FILENAME ":" FNR >errout;
      print "  Not within a recognized BEGIN/END section " >errout;
      next; 
    }
}

END {
  dump_glyph_table()
    }

###
###  We keep the following tables: 
###
###    Glyph_Names[int i] => symbolic name of glyph i
###    Glyph_Numbers[string s] => integer code of glyph
###    Flag[int i] => flag associated with glyph i (string or int)
###    Map[int i] => string for output of glyph i
###
###    Structural invariant:  
###       Glyph_Names[Glyph_Numbers[s]] = s
###       Glyph_Numbers[Glyph_Names[i]] = i 
###

function glyph_list_line() {
  ++glyph_count;
  Glyph_Names[glyph_count] = $1; 
  Glyph_Numbers[$1] = glyph_count;
}

function glyph_map_line(      code) {
  if (! ($1 in Glyph_Numbers))
    {
      print FILENAME ":" FNR " glyphmaps.awk " >errout;
      print "     Unknown glyph \"" $1 "\"" >errout;
      return;
    }
      
  if ($2 == "OMIT") 
    {
      code = Glyph_Numbers[$1];
      Flag[code] = 0;
      Map[code] = "UNUSED"; 
      return; 
    }

  if (NF < 3) 
    {
      print FILENAME ":" FNR " glyphmaps.awk " >errout;
      print "     Expected 3 fields" >errout;
      return;
    }
      
  code = Glyph_Numbers[$1];
  Flag[code] = $2; 
  Map[code] = $3; ## Need to handle quoted strings here ... 
}

function  dump_glyph_table()
{

     
  print "static struct glyph_map { int code; int flag; char *map; } "
  print "\tGlyph_Map[] = {" ; 
  print "\t{0,0,0}, /* Special case - default processing */"; 
  for (i=1; i <= glyph_count; ++i)
    {
      ## Structural invariant check
      if (Glyph_Numbers[Glyph_Names[i]] !=i) 
	{
	  print "glyphmaps.awk BUGGY, structural invariant violated" >errout;
	}

      if (! (i in Flag))
	{
	  print "Warning, glyph " Glyph_Names[i] " was not mapped " >errout;
	  printf("\t{%d, %d, %d}, /* \"%s\" MISSING  */\n", 
		 i, 0,0, Glyph_Names[i]); 
	}
      else 
	{
	  printf("\t{%d, %s, \"%s\" }, /* %s */\n", 
		 i, Flag[i], Map[i], Glyph_Names[i]); 
	}
    }
  print "}; /* End of initialized glyphmap structure */"; 
}

