/*
 *  Scan model .html file to choose typesetting conventions for 
 *  source text. 
 * Patterns in source text are represented as 
 * $$ html code $$category$$ html code $$ 
 *   where the html code may be up to 1024 characters
 *  % occuring before $ means:  Skip the remainder of this line (comment)
 *  $$ occuring before % indicates the beginning of a model pattern (see below)
 *  Other text is skipped (so that we can have a full legal html page,
 *  which may for example contain a reference to a style sheet.)
 */

/* Calling conventions: 
 *    A scanning function returns 1 + the length of the string
 *    matched (i.e., value 1 means we successfully matched an empty
 *    string); the "skipping" functions return 1 for success.  All 
 *    return 0 for failure (i.e., did not match). The pointer returned 
 *    by a successful scan is for copying into another location; the 
 *    value may be overwritten by any subsequent call. 
 *
 */

/* Buffer maintenance: 
 *    Call "Init" with the name (complete path) of a file; it opens the
 *    file and initializes the buffer.  Patterns are executed as a DFA, 
 *    with no backtracking, so we simply read a single character at a time
 *    and take the corresponding transition.  The buffer is initialized as
 *    we begin one of the scanning functions.
 */

#include "html_styles.h"
#include "groptions.h"

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>        /* On a Sparcstation running Solaris */
/* #include <strings.h> */ /* On some on other Unixes, including PC Solaris */

static FILE* file_handle = 0; 
static char *buffer; 
static int bufsize = 0;
const int initial_bufsize = 100;

static int init_scan_model( char *path ) 
{
  FILE *f; 
  int success = 0;
  f = fopen(path, "r"); 
  if (f == 0) 
    {
      fprintf(stderr,"\n*Failed to open html model file \"%s\"\n", path);
      exit(-1); 
    }
  file_handle = f; 
  buffer = malloc( initial_bufsize); 
  assert(buffer != 0); 
  bufsize = initial_bufsize; 
}

/* 
 * skip_to_model -- 
 *  Skip to occurence of $$, 
 *    return 1 if found, 0 if hit EOF instead
 */
static int skip_to_model() 
{
  int ch; /* Usually character, but could be EOF */
  int state = 1; 
  while (1) 
    {
      ch = fgetc(file_handle);
      if (ch == EOF) 
	{
	  return 0; 
	}
      switch (state) 
	{
	case 1:
	  if (ch == '$') state = 2; 
	  else if (ch == '\\' ) state = 3;
	  break;
	case 2: 
	  if (ch == '$') return 1; 
	  else if (ch == '\\') state = 3; 
	  else state = 1; 
	  break; 
	case 3: 
	  state = 1; 
	  break;
	}
    }
}

/* 
 * gather_prefix -- 
 *  Scan to occurence of $, 
 *    gathering characters along the way, and stuffing them in the 
 *    buffer.  We have a predefined limit of 1024 characters allowed
 *    in the prefix (sanity check on tag upper bound). 
 *  Return 0 on failure (e.g., eof), otherwise return 1+length of match
 */
static int gather_prefix() 
{
  int ch; /* Usually character, but could be EOF */
  int state = 4; 
  int bufpos = 0; 
  while (1) 
    {
      ch = fgetc(file_handle);
      if (ch == EOF) 
	{
	  return 0; 
	}
      switch (state) 
	{
	case 4:  /* Gather prefix characters, until $ */
	  if (ch == '$') 
	    {
	      /* End of prefix, mark the buffer and return */
	      buffer[bufpos] = '\0'; 
	      return 1+bufpos;
	    }
	  else if (ch == '\\' ) state = 5; 
	  else 
	    {
	      if (bufpos+1 >= bufsize)
		{
		  bufsize = (bufsize * 3) / 2; 
		  fprintf(stderr, "\nResizing tag buffer to %d\n", bufsize); 
		  buffer = realloc(buffer, bufsize);
		  assert(buffer != 0); 
		}
	      buffer[bufpos] = ch; 
	      ++bufpos;
	    }
	  break;
	case 5: /* Escaped character */
	  buffer[bufpos]= ch; 
	  ++bufpos;
	  state = 4; 
	  break;
	default: 
	  assert ( 0 ); 
	}
    }
}


struct style_tag { char *style_name; char *begin_tag; char *end_tag; };

/* 
 *  The table of styles is a simple linked list
 *  (which should not be a bottleneck given its intended use, but 
 *  of course this could be replaced with a decent table structure
 *  if it were used in a way that demanded better performance.) 
 */
struct style_table_cell { struct style_tag style; 
                          struct style_table_cell *link; }; 

static struct style_table_cell *style_table = 0; 

/* 
 *  We define some styles to apply by default 
 */
static struct style_tag default_styles[] = 
 {
   { "Keyword", "<B>", "</B>" },
   { "Identifier", "", "" },
   { "Declaration", 
     "<FONT SIZE=+1 COLOR=\"Red\" FACE=\"Arial,Helvetica\">", 
     "</FONT>"},
   { "Comment",   "<FONT COLOR=\"Navy\"><EM>", "</EM></FONT>"},
   { "Quoted",    "<FONT COLOR=\"Green\"><SAMP>", "</SAMP></FONT>"},
   { 0, 0, 0 } /* Marks end of table */
 }; 

char static * table_sentry = 0;           /* style entry of table sentry */
char static * empty_string = "";   /* what to use if no style */

/* 
 * Strategy: 
 *    We should never read very many different styles, so we won't bother 
 *    with good storage management.  We'll keep a simple linked list, and 
 *    always prepend to the beginning of the list. Searches are forward, 
 *    so the LAST style associated with a category is the style in effect. 
 *    We'll place the default styles at the end of the list (i.e., by pre-loading
 *    them), so they take effect in the absence of other tags. 
 */

static 
void load_default_styles() 
{
  int i =0; 
  struct style_table_cell *new_cell = 0; 

  for (i=0; default_styles[i].style_name != table_sentry; ++i) 
    {
      new_cell = malloc(sizeof(struct style_table_cell));
      assert(new_cell != 0);  /* Else we have a storage allocation error */
      new_cell->style.style_name = default_styles[i].style_name;
      new_cell->style.begin_tag = default_styles[i].begin_tag;
      new_cell->style.end_tag = default_styles[i].end_tag;
      new_cell->link = style_table;
      style_table = new_cell; 
    }
}

/*
 * Insert_model_style: 
 *    Place a new entry in the style table.  We copy references here,
 *    so it is important that the input arguments are "permanent"
 *    (it is the caller's responsibility to use strdup if necessary).
 */

static 
void insert_model_style (char *style_name, char *begin_tag, char *end_tag) 
{
  struct style_table_cell * new_cell = 0;

  new_cell = malloc(sizeof(struct style_table_cell));
  assert(new_cell != 0);  /* Else we have a storage allocation error */
  new_cell->style.style_name = style_name; 
  new_cell->style.begin_tag = begin_tag;
  new_cell->style.end_tag = end_tag;
  new_cell->link = style_table;
  style_table = new_cell; 
}
  
extern
void dump_style_table() 
{
  struct style_table_cell *ptr; 
  for (ptr = style_table; ptr != 0; ptr = ptr->link)
    {
      printf("%s::\t%s_______%s\n", 
	     ptr->style.style_name, 
	     ptr->style.begin_tag,
	     ptr->style.end_tag
	     );
    }
}

extern
void load_model_file(char *filename) 
{
  int ch = 1;  /* Char, but it could be EOF */
  int result =0; 

  /* Hold the three parts of a table entry.  These are created with "strdup"
   * so that we don't have to worry about them changing or being deallocated.
   */
  char *category = 0; 
  char *prefix = 0; 
  char *suffix = 0; 

  init_scan_model(filename);

  result = skip_to_model(); 
  while (result) 
    {
      result = gather_prefix(); 
      prefix = strdup(buffer); 

      result = gather_prefix(); 
      category = strdup(buffer); 

      result = gather_prefix();
      suffix = strdup(buffer); 

      if (result) 
	{
	  insert_model_style (category, prefix, suffix); 
	}
      if (result) result = skip_to_model(); 
    }
}

/***
***
***   The public functions interface to this package: 
***
***/

extern void init_html_styles() 
{
  load_default_styles(); 
  if (optblock.model_file_name) 
    {
      load_model_file(optblock.model_file_name); 
    }
}


/* 
 * style map:
 *  Look up symbolic style name.  This is not implemented efficiently,
 *  so it is best to call it once for each desired style name and store
 *  styles in local variables. 
 *  
 *  Two  values are passed by reference, the beginning and 
 *  ending tags sequence for the given symbolic style. These should
 *  be references to strings. If everything is ok, return value is
 *  1.  If style is not found, return value is 0, and 
 */
extern int  html_style_map ( char *style_name, 
			     char * * begin_tag, 
			     char * * end_tag )
{
  struct style_table_cell *ptr = 0;

  for (ptr= style_table; ptr != 0; ptr=ptr->link)
    {
      if (strcasecmp(style_name, ptr->style.style_name)==0) 
	{
	  * begin_tag = ptr->style.begin_tag;  /* Reference! */
	  * end_tag = ptr->style.end_tag;      /* Reference! */
	  return 1; 
	}
    }
  * begin_tag = empty_string;
  * end_tag = empty_string;
  return 0;
}      

