/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.photran.errorparsers.xlf;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.cdt.core.ErrorParserManager;
import org.eclipse.cdt.core.IErrorParser;
import org.eclipse.cdt.core.IMarkerGenerator;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;

/**
 * Error parser for the IBM XLF compiler
 * 
 * @author Craig Rasmussen
 */
public class XLFCompilerErrorParser implements IErrorParser
{
	Pattern p1 = Pattern.compile("^\"([^\"]*)\", line (\\d+)\\.(\\d+): (\\d+)-(\\d+) \\(([USEWLI])\\) (.*)$");
	Pattern p2 = Pattern.compile("^\"([^\"]*)\", (\\d+)-(\\d+) \\(([USEWLI])\\) (.*)$");
	private String fileName;
	private String lineNum;
	private String level;
	private String message;
	private int num;
		
	public XLFCompilerErrorParser()
	{
		fileName = null;
		lineNum = null;
		level = "S";
		message = null;
		num = -1;
	}
	
	/**
	 * This function returns the file name extracted from
	 * the error message.
	 * @return The string value of the given file name.
	 */
	public String getFileName()
	{
		return fileName;
	}
	
	/**
	 * This function returns the line number of
	 * the error.
	 * @return The integer value of the line number.
	 */
	
	public int getLineNumber()
	{
		return num;
	}
	
	/**
	 * This function returns the severity of the
	 * error that has occured.
	 * @return The string value of the severity.
	 */
	public String getSeverity()
	{
		return level;
	}
	
	/**
	 * This function returns the descriptive string of the
	 * error that has occured.
	 * @return The string value of the message.
	 */
	public String getMessage()
	{
		return message;
	}
	
	/**
	 * This function parses the error message occured and fills the
	 * class variables fileName, lineNum, message, level( severity ).
	 * @param line is the error message generated by the xlF compiler. 
	 * @return a boolean value indicating the success/failure of 
	 * extracting the values for fileName, lineNum, message, level. 
	 */
	
	public boolean parseLine(String line)
	{
		String desc = null;
		Matcher m = p1.matcher(line);
        if (m.matches())
        {
            fileName = m.group(1);
            lineNum = m.group(2);
            m.group(3);
            m.group(4);
            m.group(5);
            level = m.group(6);
            desc = m.group(7);
            message = desc;
            try
            {
                num = Integer.parseInt(lineNum);
            }
            catch (NumberFormatException e)
            {
                throw e;
            }
        }
        else
        {
            m = p2.matcher(line);
            if (m.matches())
            {
                fileName = m.group(1);
                m.group(2);
                m.group(3);
                level = m.group(4);
                desc = m.group(5);
                message = desc;
            }
            else
            {
                return false;
            }
        }
        return true;
	}
	
	
	
	
	public boolean processLine(String line, ErrorParserManager eoParser) {
		return processLine(line, eoParser, IMarkerGenerator.SEVERITY_ERROR_RESOURCE);
	}

	public boolean processLine(String line, ErrorParserManager eoParser, int inheritedSeverity) {
		// XLF error messsage format:
		//
		// "<filename>", line <lineno>.<column>: 15<CC>-<NNN> (<S>) <description>
		// "<filename>", line <lineno>.<column>: 15<CC>-<NNN> <description>
		//
		// <CC> is one of:
		//      00    : Indicates a code generation or optimization message.
		//      01    : Indicates an XL Fortran common message.
		//      11-20 : Indicates a Fortran-specific message.
		//      25    : Indicates a run-time message from an XL Fortran application program.
		//      85    : Indicates a loop-transformation message.
		//      86    : Indicates an interprocedural analysis (IPA) message.
		//
		// <NNN> is the message number
		//
		// <S> is one of:
		//      U     : An unrecoverable error.
		//      S     : A severe error.
		//      E     : An error that the compiler can correct.
		//      W     : Warning message. 
		//      L     : Language conformance warning message.
		//      I     : Informational message.
		
		if( parseLine(line) == false )
		{
			return false;
		}
        if (!Path.EMPTY.isValidPath(fileName)) { return false; }

        IFile file = eoParser.findFileName(fileName);
        if (file != null)
        {
             if (eoParser.isConflictingName(fileName)) 
            {
                message = "[Conflicting names: " + fileName + " ] " + message; //$NON-NLS-1$ //$NON-NLS-2$
                file = null;
            }
        }
        else
        {
            file = eoParser.findFilePath(fileName);
            if (file == null)
            {
                // one last try before bailing out we may be in a wrong
                // directory. This will happen, for example in the Makefile:
                // all: foo.c
                // cd src3; gcc -c bar/foo.c
                // the user do a cd(1).
                IPath path = new Path(fileName);
                if (path.segmentCount() > 1)
                {
                    String name = path.lastSegment();
                    file = eoParser.findFileName(fileName);
                    if (file != null)
                    {
                        if (eoParser.isConflictingName(fileName))
                        {
                            message = "[Conflicting names: " + name + " ] " + message; //$NON-NLS-1$ //$NON-NLS-2$
                            file = null;
                        }
                    }
                }
            }
        }

        // Display the fileName.
        if (file == null)
        {
            message = message + " [" + fileName + "]"; //$NON-NLS-1$ //$NON-NLS-2$
        }

        int severity = extractSeverity(level, inheritedSeverity);

        eoParser.generateMarker(file, num, message, severity, null);
        return true;
    }

    private int extractSeverity(String desc, int defaultSeverity)
    {
        int severity = defaultSeverity;

        if (desc.equals("U") || desc.equals("S") )
        {
            severity = IMarkerGenerator.SEVERITY_ERROR_BUILD;
        }
        else if (desc.equals("E"))
        {
            severity = IMarkerGenerator.SEVERITY_ERROR_RESOURCE;
        }
        else if (desc.equals("W") || desc.equals("L") )
        {
            severity = IMarkerGenerator.SEVERITY_WARNING;
        }
        else if (desc.equals("I"))
        {
            severity = IMarkerGenerator.SEVERITY_INFO;
        }
        return severity;
    }
}
