/*
 * MIT License
 *
 * Copyright (c) 2023 University of Oregon, Kevin Huck
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#pragma once

#include "zerosum.h"
#include "nvml.h"

#define NVML_CALL(call)                                                      \
do {                                                                         \
    nvmlReturn_t _status = call;                                             \
    if (_status != NVML_SUCCESS) {                                           \
        const char *errstr = nvmlErrorString(_status);                       \
        fprintf(stderr, "%s:%d: error: function %s failed with error %d: %s.\n", \
                __FILE__, __LINE__, #call, _status, errstr);                 \
        exit(-1);                                                            \
    }                                                                        \
} while (0);

#define MILLIONTH 1.0/(1024.0*1024.0) // scale to MB
#define BILLIONTH 1.0/(1024.0*1024.0*1024.0) // scale to GB
#define PCIE_THROUGHPUT 1.0/1024.0  // to scale KB to MB
#define NVLINK_BW 1.0/1024.0  // to scale MB/s to GB/s
#define WATTS 1.0e-3  // scale mW to W

namespace zerosum {

std::vector<nvmlDevice_t> initialize_nvml(void);
void finalize_nvml(void);

class ScopedNVML {
private:
public:
    std::vector<nvmlDevice_t> devices;
    ScopedNVML(void) {
        devices = initialize_nvml();
    }
    ~ScopedNVML(void) {
        finalize_nvml();
    }
};

}
