/*
 * Copyright (c) 2014-2021 Kevin Huck
 * Copyright (c) 2014-2021 University of Oregon
 *
 * Distributed under the Boost Software License, Version 1.0. (See accompanying
 * file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
 * For OMPT target offload prototype, see https://github.com/ROCm-Developer-Tools/aomp/tree/aomp-dev/examples/tools/ompt/veccopy-ompt-target-tracing
 */

#include <omp-tools.h>
#include <unordered_map>
#include "string.h"
#include "stdio.h"
#include "apex_api.hpp"
#include "apex_types.h"
#include "thread_instance.hpp"
#include "apex_cxx_shared_lock.hpp"
#include <atomic>
#include <memory>
#include <string>
#include "apex_assert.h"
#include "inttypes.h"
#include "event_listener.hpp"
#include "async_thread_node.hpp"
#include "apex.hpp"
#if defined(APEX_WITH_PERFETTO)
#include "perfetto_listener.hpp"
#endif
#include "trace_event_listener.hpp"
#ifdef APEX_HAVE_OTF2
#include "otf2_listener.hpp"
#endif
//#include "apex_rocm_smi.hpp"


std::mutex apex_apex_threadid_mutex;
std::atomic<uint64_t> apex_numthreads{0};
APEX_NATIVE_TLS int64_t apex_threadid{-1};
static std::atomic<bool> enabled{false};

constexpr size_t apex_ompt_buffer_request_size{16*1024};

class linked_timer {
    public:
        void * prev;
        std::shared_ptr<apex::task_wrapper> tw;
        bool timing;
        const void * codeptr; // for implicit tasks
        inline void start(void) { apex::start(tw); timing = true;  }
        inline void yield(void) { apex::yield(tw); timing = false; }
        inline void stop(void)  { apex::stop(tw);  timing = false; }
        /* constructor */
        linked_timer(const char * name,
            uint64_t task_id,
            void *p,
            std::shared_ptr<apex::task_wrapper> &parent,
            bool auto_start,
            const void * codeptr_ra,
            bool is_par_reg) :
            prev(p), timing(auto_start), codeptr(codeptr_ra) {
            // No GUIDs generated by the runtime? Generate our own.
            if (task_id == 0ULL) {
                tw = apex::new_task(name, UINT64_MAX, parent);
            } else {
                tw = apex::new_task(name, task_id, parent);
            }
            if (tw != nullptr) {
                if (is_par_reg) { tw->explicit_trace_start = true; }
                if (auto_start) { this->start(); }
            } else {
                timing = false;
            }
        }
        /* destructor */
        ~linked_timer() {
            if (timing) {
                apex::stop(tw);
            }
        }
};

/* This class is necessary so we can clean up before our globals are destroyed at exit */
class OmptGlobals{
private:
    OmptGlobals() : deltaTimestamp(0) {} // Disallow instantiation outside of the class.
    // Timestamp at trace initialization time. Used to normalized other
    // timestamps
    int64_t deltaTimestamp;
    uint64_t cpuTimestamp;
    std::mutex correlation_map_mutex;
    std::unordered_map<uint32_t, std::shared_ptr<apex::task_wrapper>> timer_map;
    std::unordered_map<uint32_t, apex::async_event_data> data_map;
    std::unordered_map<int, ompt_device_t*> device_map;
public:
    OmptGlobals(const OmptGlobals&) = delete;
    OmptGlobals& operator=(const OmptGlobals &) = delete;
    OmptGlobals(OmptGlobals &&) = delete;
    OmptGlobals & operator=(OmptGlobals &&) = delete;
    // if our globals are destroyed before APEX can finalize, then finalize now! */
    ~OmptGlobals() {
        apex::finalize();
    }

    static auto& instance(){
        static OmptGlobals test;
        return test;
    }
    static int64_t& delta() {
        return instance().deltaTimestamp;
    }
    static bool set_delta_base() {
        instance().cpuTimestamp = apex::profiler::now_ns();
        return true;
    }
    static bool set_delta(uint64_t gpu) {
        instance().deltaTimestamp = instance().cpuTimestamp - gpu;
        return true;
    }
    void lock() { correlation_map_mutex.lock(); }
    void unlock() { correlation_map_mutex.unlock(); }

    static void insert_timer(uint32_t id, std::shared_ptr<apex::task_wrapper> timer) {
        instance().correlation_map_mutex.lock();
        instance().timer_map[id] = timer;
        //std::cerr << "adding parent task_wrapper: " << id << std::endl;
        instance().correlation_map_mutex.unlock();
    }

    static std::shared_ptr<apex::task_wrapper> find_timer(uint32_t id) {
        std::shared_ptr<apex::task_wrapper> timer = nullptr;
        OmptGlobals& g = instance();
        g.lock();
        auto iter = g.timer_map.find(id);
        if (iter != g.timer_map.end()) {
            timer = iter->second;
        } else {
            //std::cerr << "parent task_wrapper not found: " << id << std::endl;
        }
        g.unlock();
        return timer;
    }

    static void erase_timer(uint32_t id) {
        OmptGlobals& g = instance();
        g.lock();
        g.timer_map.erase(id);
        g.unlock();
    }

    static void insert_data(uint32_t id, apex::async_event_data& data) {
        instance().correlation_map_mutex.lock();
        instance().data_map[id] = data;
        instance().correlation_map_mutex.unlock();
    }

    static void update_data(uint32_t id) {
        OmptGlobals& g = instance();
        g.lock();
        auto iter = g.data_map.find(id);
        if (iter != g.data_map.end()) {
            iter->second.parent_ts_stop = apex::profiler::now_ns();
            // std::cout << "Updated " << id << std::endl;
        }
        g.unlock();
    }

    static apex::async_event_data find_data(uint32_t id) {
        apex::async_event_data as_data;
        OmptGlobals& g = instance();
        g.lock();
        auto iter = g.data_map.find(id);
        if (iter != g.data_map.end()) {
            as_data = iter->second;
        } else {
            //std::cerr << "target data not found: " << id << std::endl;
        }
        g.data_map.erase(id);
        g.unlock();
        return as_data;
    }

    static void insert_device(int index, ompt_device_t *device) {
        instance().lock();
        instance().device_map.insert(std::pair<int, ompt_device_t*>(index, device));
        instance().unlock();
    }

    static ompt_device_t* get_device(int index) {
        instance().lock();
        auto d = instance().device_map[index];
        instance().unlock();
        return d;
    }
};

// forward declaration
static uint64_t apex_ompt_translate_time(int index, ompt_device_time_t time);

std::shared_ptr<apex::task_wrapper> start_async_task(const std::string &name, uint32_t correlationId, long unsigned int& parent_thread) {
    apex::in_apex prevent_deadlocks;
    // get the parent GUID, then erase the correlation from the map
    std::shared_ptr<apex::task_wrapper> parent = nullptr;
    if (correlationId > 0) {
        parent = OmptGlobals::find_timer(correlationId);
        if (parent != nullptr) {
            parent_thread = parent->thread_id;
        }
    }
    // create a task_wrapper, as a GPU child of the parent on the CPU side
    std::shared_ptr<apex::task_wrapper> tt = apex::new_task(name, UINT64_MAX, parent);
    return tt;
}

void stop_async_task(std::shared_ptr<apex::task_wrapper> tt, uint64_t start, uint64_t end,
    uint32_t correlationId, apex::ompt_thread_node &node) {
    // Handle tracing, if necessary
    apex::async_event_data as_data;
    if (correlationId > 0) {
        as_data = OmptGlobals::find_data(correlationId);
    }
    // create an APEX profiler to store this data - we can't start
    // then stop because we have timestamps already.
    auto prof = std::make_shared<apex::profiler>(tt);
    bool do_trace{true};
    if (start == 0 || end == 0) {
        prof->set_start(as_data.parent_ts_start);
        prof->set_end(as_data.parent_ts_stop);
        do_trace = false;
    } else {
        prof->set_start(start);
        prof->set_end(end);
    }
    //std::cout << "Parent prof " << correlationId << ": " << (uint64_t)as_data.parent_ts_start << " " << (uint64_t)as_data.parent_ts_stop << std::endl;
    //std::cout << "Final prof " << correlationId << ": " << prof->get_start_ns() << " " << prof->get_stop_ns() << std::endl;
    // important!  Otherwise we might get the wrong end timestamp.
    prof->stopped = true;
    // Get the singleton APEX instance
    static apex::apex* instance = apex::apex::instance();
    // fake out the profiler_listener
    instance->the_profiler_listener->push_profiler_public(prof);
    if (do_trace) {
#if defined(APEX_WITH_PERFETTO)
    if (apex::apex_options::use_perfetto()) {
        apex::perfetto_listener * tel =
            (apex::perfetto_listener*)instance->the_perfetto_listener;
        as_data.cat = "ControlFlow";
        as_data.reverse_flow = false;
        tel->on_async_event(node, prof, as_data);
    }
#endif
    if (apex::apex_options::use_trace_event()) {
        apex::trace_event_listener * tel =
            (apex::trace_event_listener*)instance->the_trace_event_listener;
        as_data.cat = "ControlFlow";
        as_data.reverse_flow = false;
        tel->on_async_event(node, prof, as_data);
    }
#ifdef APEX_HAVE_OTF2
    if (apex::apex_options::use_otf2()) {
        apex::otf2_listener * tol =
            (apex::otf2_listener*)instance->the_otf2_listener;
        tol->on_async_event(node, prof);
    }
#endif
    }
    // have the listeners handle the end of this task
    instance->complete_task(tt);
}

void store_profiler_data(const std::string &name,
        uint64_t start, uint64_t end, apex::ompt_thread_node &node,
        std::shared_ptr<apex::task_wrapper> parent, bool otf2_trace = true) {
    apex::in_apex prevent_deadlocks;
    apex::async_event_data as_data;
    as_data.flow = false;
    // create a task_wrapper, as a GPU child of the parent on the CPU side
    auto tt = apex::new_task(name, UINT64_MAX, parent);
    // create an APEX profiler to store this data - we can't start
    // then stop because we have timestamps already.
    auto prof = std::make_shared<apex::profiler>(tt);
    prof->set_start(start);
    prof->set_end(end);
    // important!  Otherwise we might get the wrong end timestamp.
    prof->stopped = true;
    // Get the singleton APEX instance
    static apex::apex* instance = apex::apex::instance();
    // fake out the profiler_listener
    instance->the_profiler_listener->push_profiler_public(prof);
    // Handle tracing, if necessary
#if defined(APEX_WITH_PERFETTO)
    if (apex::apex_options::use_perfetto()) {
        apex::perfetto_listener * tel =
            (apex::perfetto_listener*)instance->the_perfetto_listener;
        tel->on_async_event(node, prof, as_data);
    }
#endif
    if (apex::apex_options::use_trace_event()) {
        apex::trace_event_listener * tel =
            (apex::trace_event_listener*)instance->the_trace_event_listener;
        tel->on_async_event(node, prof, as_data);
    }
#ifdef APEX_HAVE_OTF2
    if (apex::apex_options::use_otf2() && otf2_trace) {
        apex::otf2_listener * tol =
            (apex::otf2_listener*)instance->the_otf2_listener;
        tol->on_async_event(node, prof);
    }
#else
    APEX_UNUSED(otf2_trace);
#endif
    // have the listeners handle the end of this task
    instance->complete_task(tt);
}

/* Handle counters from asynchronous activity */
void store_counter_data(const char * name, const std::string& ctx,
    uint64_t end, double value, apex::ompt_thread_node &node) {
    apex::in_apex prevent_deadlocks;
    std::stringstream ss;
    if (name == nullptr) {
        ss << "GPU: " << ctx;
    } else {
        ss << "GPU: " << name << " " << ctx;
    }
    std::string tmp{ss.str()};
    auto task_id = apex::task_identifier::get_task_id(tmp);
    std::shared_ptr<apex::profiler> prof =
        std::make_shared<apex::profiler>(task_id, value);
    prof->is_counter = true;
    prof->set_end(end);
    // Get the singleton APEX instance
    static apex::apex* instance = apex::apex::instance();
    // fake out the profiler_listener
    instance->the_profiler_listener->push_profiler_public(prof);
    // Handle tracing, if necessary
#if defined(APEX_WITH_PERFETTO)
    if (apex::apex_options::use_perfetto()) {
        apex::perfetto_listener * tel =
            (apex::perfetto_listener*)instance->the_perfetto_listener;
        tel->on_async_metric(node, prof);
    }
#endif
    if (apex::apex_options::use_trace_event()) {
        apex::trace_event_listener * tel =
            (apex::trace_event_listener*)instance->the_trace_event_listener;
        tel->on_async_metric(node, prof);
    }
#ifdef APEX_HAVE_OTF2
    if (apex::apex_options::use_otf2()) {
        apex::otf2_listener * tol =
            (apex::otf2_listener*)instance->the_otf2_listener;
        tol->on_async_metric(node, prof);
    }
#endif
}

void store_counter_data(const char * name, const std::string& ctx,
    uint64_t end, size_t value, apex::ompt_thread_node &node) {
    store_counter_data(name, ctx, end, (double)(value), node);
}

// Simple print routine that this example uses while traversing
// through the trace records returned as part of the buffer-completion callback
static void print_record_ompt(ompt_record_ompt_t *rec) {
  if (rec == NULL) return;

  DEBUG_PRINT("rec=%p type=%d time=%lu thread_id=%lu target_id=%lu\n",
     (void*)rec, rec->type, rec->time, rec->thread_id, rec->target_id);

  static std::unordered_map<ompt_id_t,const void*> active_target_addrs;
  static std::unordered_map<ompt_id_t,int> active_target_devices;
  static std::unordered_map<uint32_t, std::shared_ptr<apex::task_wrapper>> target_map;
  static std::unordered_map<uint32_t, uint64_t> target_start_times;
  static std::unordered_map<uint32_t, uint64_t> target_end_times;
  static std::unordered_map<uint32_t, uint64_t> target_parent_thread_ids;
  static std::mutex target_lock;
  long unsigned int parent_thread = 0;

  switch (rec->type) {
  case ompt_callback_target:
#if defined(APEX_HAVE_OMPT_5_1)
  case ompt_callback_target_emi:
#endif
    {
      ompt_record_target_t target_rec = rec->record.target;
      DEBUG_PRINT("\tRecord Target: kind=%d endpoint=%d device=%d task_id=%lu target_id=%lu codeptr=%p\n",
         target_rec.kind, target_rec.endpoint, target_rec.device_num,
         target_rec.task_id, target_rec.target_id, target_rec.codeptr_ra);
         if (target_rec.endpoint == ompt_scope_begin) {
            std::stringstream ss;
            ss << "GPU: OpenMP Target";
#if defined(APEX_HAVE_OMPT_5_1)
            if(rec->type == ompt_callback_target_emi) ss << " EMI";
#endif
            if (target_rec.codeptr_ra != nullptr) {
                ss << ": UNRESOLVED ADDR " << target_rec.codeptr_ra;
            }
            std::string name{ss.str()};
               auto tt = start_async_task(name, rec->target_id, parent_thread);
            std::unique_lock<std::mutex> l(target_lock);
            //if (rec->time > 0) {
                target_map[rec->target_id] = tt;
            //}
            target_start_times[rec->target_id] = rec->time;
            active_target_addrs[rec->target_id] = target_rec.codeptr_ra;
            active_target_devices[rec->target_id] = target_rec.device_num;
            target_parent_thread_ids[rec->target_id] = parent_thread;
         } else if (target_rec.endpoint == ompt_scope_end) {
            std::shared_ptr<apex::task_wrapper> tt;
            uint64_t start;
            uint64_t end;
            {
                std::unique_lock<std::mutex> l(target_lock);
                tt = target_map[rec->target_id];
                // the AMD runtime no longer (versions 5.1-5.5) gives times for the
                // target event. So, we'll use the times from the inner events
                // as the start/stop time for this event in case it's needed.
                start = target_start_times[rec->target_id];
                end = rec->time;
                if (end == 0) {
                    if (target_end_times.count(rec->target_id) > 0) {
                        end = target_end_times[rec->target_id] + 1000;
                    }
                }
                // std::cout << "Target " << rec->target_id << ": " << start << " - " << end << std::endl;
                parent_thread = target_parent_thread_ids[rec->target_id];
                active_target_addrs.erase(rec->target_id);
                active_target_devices.erase(rec->target_id);
                target_map.erase(rec->target_id);
                target_start_times.erase(rec->target_id);
                target_end_times.erase(rec->target_id);
                target_parent_thread_ids.erase(rec->target_id);
            }
            /* If we have a target region with a device id of -1, we might not get
               a target region start event - so ignore this end event for now. */
            if (tt != nullptr) {
                apex::ompt_thread_node node(target_rec.device_num, parent_thread, APEX_ASYNC_KERNEL);
                if (start > 0 && end > 0) {
                       stop_async_task(tt,
                        apex_ompt_translate_time(target_rec.device_num, start),
                        apex_ompt_translate_time(target_rec.device_num, end),
                        rec->target_id, node);
                }
                OmptGlobals::erase_timer(rec->target_id);
            }
         }
      break;
    }
  case ompt_callback_target_data_op:
#if defined(APEX_HAVE_OMPT_5_1)
  case ompt_callback_target_data_op_emi:
#endif
    {
      ompt_record_target_data_op_t target_data_op_rec = rec->record.target_data_op;
      DEBUG_PRINT("\tRecord DataOp: host_op_id=%lu optype=%d src_addr=%p src_device=%d "
         "dest_addr=%p dest_device=%d bytes=%lu end_time=%lu duration=%lu ns codeptr=%p\n",
         target_data_op_rec.host_op_id, target_data_op_rec.optype,
         target_data_op_rec.src_addr, target_data_op_rec.src_device_num,
         target_data_op_rec.dest_addr, target_data_op_rec.dest_device_num,
         target_data_op_rec.bytes, target_data_op_rec.end_time,
         target_data_op_rec.end_time - rec->time,
         target_data_op_rec.codeptr_ra);
        int time_index = target_data_op_rec.dest_device_num;
            std::stringstream ss;
            ss << "GPU: OpenMP Target DataOp";
#if defined(APEX_HAVE_OMPT_5_1)
            if(rec->type == ompt_callback_target_data_op_emi) ss << " EMI";
#endif
            switch (target_data_op_rec.optype) {
                case ompt_target_data_alloc: {
                    ss << " Alloc";
                    break;
                }
                case ompt_target_data_transfer_to_device: {
                    ss << " Xfer to Dev";
                    break;
                }
                case ompt_target_data_transfer_from_device: {
                    ss << " Xfer from Dev";
                    time_index = target_data_op_rec.src_device_num;
                    break;
                }
                case ompt_target_data_delete: {
                    ss << " Delete";
                    break;
                }
                case ompt_target_data_associate: {
                    ss << " Associate";
                    break;
                }
                case ompt_target_data_disassociate: {
                    ss << " Disassociate";
                    break;
                }
#if defined(APEX_HAVE_OMPT_5_1)
                case ompt_target_data_alloc_async: {
                    ss << " Alloc";
                    break;
                }
                case ompt_target_data_transfer_to_device_async: {
                    ss << " Xfer to Dev Async";
                    break;
                }
                case ompt_target_data_transfer_from_device_async: {
                    ss << " Xfer from Dev Async";
                    time_index = target_data_op_rec.src_device_num;
                    break;
                }
                case ompt_target_data_delete_async: {
                    ss << " Delete Async";
                    break;
                }
#endif
            }
            std::shared_ptr<apex::task_wrapper> tt;

            const void* codeptr_ra;
            {
                std::unique_lock<std::mutex> l(target_lock);
                // if we have a legit GPU target event, use it
                if (target_map.count(rec->target_id) > 0) {
                    tt = target_map[rec->target_id];
                    // otherwise, use the host side target call
                } else {
                    tt = OmptGlobals::find_timer(rec->target_id);
                }
                codeptr_ra = active_target_addrs[rec->target_id];
                parent_thread = target_parent_thread_ids[rec->target_id];
                // the AMD runtime no longer gives times for the
                // target event. So, we'll use this time, and save
                // the stop time for this event in case it's needed.
                if (target_start_times[rec->target_id] == 0) {
                    target_start_times[rec->target_id] = rec->time - 1000;
                }
                target_end_times[rec->target_id] = target_data_op_rec.end_time;
                //std::cout << "Updated Target " << rec->target_id << ": " << target_start_times[rec->target_id] << " - " << target_end_times[rec->target_id] << std::endl;
            }
            if (codeptr_ra != nullptr) {
                ss << ": UNRESOLVED ADDR " << codeptr_ra;
            }
            std::string name{ss.str()};
            apex::ompt_thread_node node(target_data_op_rec.dest_device_num,
                parent_thread, APEX_ASYNC_MEMORY);
            store_profiler_data(name,
                apex_ompt_translate_time(time_index, rec->time),
                apex_ompt_translate_time(time_index, target_data_op_rec.end_time),
                node, tt);
            store_counter_data("OpenMP Target DataOp", "Bytes",
                apex_ompt_translate_time(time_index, target_data_op_rec.end_time),
                target_data_op_rec.bytes, node);
            // converting from B/us to MB/s
            double bw = (target_data_op_rec.bytes) / (target_data_op_rec.end_time - rec->time);
            store_counter_data("OpenMP Target DataOp", "BW (MB/s)",
                apex_ompt_translate_time(time_index, target_data_op_rec.end_time),
                bw, node);
      break;
    }
  case ompt_callback_target_submit:
#if defined(APEX_HAVE_OMPT_5_1)
  case ompt_callback_target_submit_emi:
#endif
    {
      ompt_record_target_kernel_t target_kernel_rec = rec->record.target_kernel;
      DEBUG_PRINT("\tRecord Submit: host_op_id=%lu requested_num_teams=%u granted_num_teams=%u "
         "end_time=%lu duration=%lu ns\n",
         target_kernel_rec.host_op_id, target_kernel_rec.requested_num_teams,
         target_kernel_rec.granted_num_teams, target_kernel_rec.end_time,
         target_kernel_rec.end_time - rec->time);
            std::stringstream ss;
            ss << "GPU: OpenMP Target Submit";
#if defined(APEX_HAVE_OMPT_5_1)
            if(rec->type == ompt_callback_target_submit_emi) ss << " EMI";
#endif
            int device_num;
            std::shared_ptr<apex::task_wrapper> tt;
            const void* codeptr_ra;
            {
                std::unique_lock<std::mutex> l(target_lock);
                // if we have a legit GPU target event, use it
                if (target_map.count(rec->target_id) > 0) {
                    tt = target_map[rec->target_id];
                    // otherwise, use the host side target call
                } else {
                    tt = OmptGlobals::find_timer(rec->target_id);
                }
                codeptr_ra = active_target_addrs[rec->target_id];
                device_num = active_target_devices[rec->target_id];
                parent_thread = target_parent_thread_ids[rec->target_id];
                if (target_start_times[rec->target_id] == 0) {
                    target_start_times[rec->target_id] = rec->time - 1000;
                }
                target_end_times[rec->target_id] = target_kernel_rec.end_time;
                //std::cout << "Updated Target " << rec->target_id << ": " << target_start_times[rec->target_id] << " - " << target_end_times[rec->target_id] << std::endl;
            }
            if (codeptr_ra != nullptr) {
                ss << ": UNRESOLVED ADDR " << codeptr_ra;
            }
            std::string name{ss.str()};
            apex::ompt_thread_node node(device_num, parent_thread,
                APEX_ASYNC_KERNEL);
            store_profiler_data(name,
                apex_ompt_translate_time(device_num, rec->time),
                apex_ompt_translate_time(device_num, target_kernel_rec.end_time),
                node, tt);
    break;
    }
  default:
    APEX_ASSERT(false);
    break;
  }
}

// Deallocation routine that will be called by the tool when a buffer
// previously allocated by the buffer-request callback is no longer required.
// The deallocation method must match the allocation routine. Here
// free is used for corresponding malloc
static void delete_buffer_ompt(ompt_buffer_t *buffer) {
  free(buffer);
  DEBUG_PRINT("Deallocated %p\n", buffer);
}

/* Function pointers.  These are all queried from the runtime during
 *  * ompt_initialize() */
static ompt_set_callback_t ompt_set_callback = nullptr;
static ompt_finalize_tool_t ompt_finalize_tool = nullptr;
/*
static ompt_enumerate_states_t ompt_enumerate_states;
static ompt_enumerate_mutex_impls_t ompt_enumerate_mutex_impls;
static ompt_get_callback_t ompt_get_callback;
static ompt_get_thread_data_t ompt_get_thread_data;
static ompt_get_num_procs_t ompt_get_num_procs;
static ompt_get_num_places_t ompt_get_num_places;
static ompt_get_place_proc_ids_t ompt_get_place_proc_ids;
static ompt_get_place_num_t ompt_get_place_num;
static ompt_get_partition_place_nums_t ompt_get_partition_place_nums;
static ompt_get_proc_id_t ompt_get_proc_id;
static ompt_get_parallel_info_t ompt_get_parallel_info;
static ompt_get_task_info_t ompt_get_task_info;
static ompt_get_task_memory_t ompt_get_task_memory;
static ompt_get_target_info_t ompt_get_target_info;
static ompt_get_num_devices_t ompt_get_num_devices;
static ompt_get_unique_id_t ompt_get_unique_id;
static ompt_function_lookup_t ompt_function_lookup;
*/
//static ompt_get_device_num_procs_t ompt_get_device_num_procs = nullptr;
static ompt_get_device_time_t ompt_get_device_time = nullptr;
static ompt_translate_time_t ompt_translate_time = nullptr;
static ompt_set_trace_ompt_t ompt_set_trace_ompt = nullptr;
//static ompt_set_trace_native_t
static ompt_start_trace_t ompt_start_trace = nullptr;
//static ompt_pause_trace_t
static ompt_flush_trace_t ompt_flush_trace = nullptr;
static ompt_stop_trace_t ompt_stop_trace = nullptr;
static ompt_get_record_ompt_t ompt_get_record_ompt = nullptr;
static ompt_advance_buffer_cursor_t ompt_advance_buffer_cursor = nullptr;
//static ompt_get_record_type = nullptr;
//static ompt_get_record_native = nullptr;
//static ompt_get_record_abstract = nullptr;

// forward declare so we can stop the initial task when forced to shut down
void apex_ompt_stop(ompt_data_t * ompt_data);

static ompt_data_t * the_initial_task = nullptr;

namespace apex {

} // end apex namespace

/* These methods are some helper functions for starting/stopping timers */

void apex_ompt_start(const char * state,
        ompt_data_t * ompt_data,
        ompt_data_t * region_data,
        bool auto_start,
        const void * codeptr_ra = nullptr,
        bool is_par_reg = false) {
    if (!enabled) { return; }
    static std::shared_ptr<apex::task_wrapper> nothing(nullptr);
    linked_timer* tmp;
    /* First check if there's a parent "region" - could be a task */
    if (ompt_data->ptr == nullptr && region_data != nullptr) {
        /* Get the parent scoped timer */
        linked_timer* parent = (linked_timer*)(region_data->ptr);
        if (parent != nullptr) {
            tmp = new linked_timer(state, ompt_data->value, ompt_data->ptr,
            parent->tw, auto_start, codeptr_ra, is_par_reg);
        } else {
            tmp = new linked_timer(state, ompt_data->value, ompt_data->ptr,
            nothing, auto_start, codeptr_ra, is_par_reg);
        }
#if 0
    /* if the ompt_data->ptr pointer is not null, that means we have an implicit
     * parent and there's no need to specify a parent */
    } else if (ompt_data->ptr != nullptr) {
        /* Get the parent scoped timer */
        linked_timer* previous = (linked_timer*)(ompt_data->ptr);
        tmp = new linked_timer(state, ompt_data->value, ompt_data->ptr,
        previous->tw, true, codeptr_ra);
#endif
    } else {
        tmp = new linked_timer(state, ompt_data->value, ompt_data->ptr,
        nothing, auto_start, codeptr_ra, is_par_reg);
    }

    /* Save the address of the scoped timer with the parallel region
     * or task, so we can stop the timer later */
    ompt_data->ptr = (void*)(tmp);
}

void apex_ompt_stop(ompt_data_t * ompt_data) {
    if (!enabled) { return; }
    APEX_ASSERT(ompt_data->ptr);
    void* tmp = ((linked_timer*)(ompt_data->ptr))->prev;
    delete((linked_timer*)(ompt_data->ptr));
    ompt_data->ptr = tmp;
}

/*
 * Mandatory Events
 *
 * The following events are supported by all OMPT implementations.
 */

/* Event #1, thread begin */
extern "C" void apex_thread_begin(
    ompt_thread_t thread_type,   /* type of thread */
    ompt_data_t *thread_data     /* data of thread */)
{
    if (!enabled) { return; }
    APEX_UNUSED(thread_data);
    {
        std::unique_lock<std::mutex> l(apex_apex_threadid_mutex);
        apex_threadid = apex_numthreads++;
    }
    switch (thread_type) {
        case ompt_thread_initial:
            apex::register_thread("OpenMP Initial Thread");
            //apex::sample_value("OpenMP Initial Thread", 1);
            break;
        case ompt_thread_worker:
            apex::register_thread("OpenMP Worker Thread");
            //apex::sample_value("OpenMP Worker Thread", 1);
            break;
        case ompt_thread_other:
            apex::register_thread("OpenMP Other Thread");
            //apex::sample_value("OpenMP Other Thread", 1);
            break;
        case ompt_thread_unknown:
        default:
            apex::register_thread("OpenMP Unknown Thread");
            //apex::sample_value("OpenMP Unknown Thread", 1);
    }
    DEBUG_PRINT("New %d thread\n", thread_type);
}

/* Event #2, thread end */
extern "C" void apex_thread_end(
    ompt_data_t *thread_data              /* data of thread                      */
) {
    if (!enabled) { return; }
    APEX_UNUSED(thread_data);
    apex::exit_thread();
}

/* Event #3, parallel region begin */
static void apex_parallel_region_begin (
    ompt_data_t *encountering_task_data,         /* data of encountering task           */
    const ompt_frame_t *encountering_task_frame,  /* frame data of encountering task     */
    ompt_data_t *parallel_data,                  /* data of parallel region             */
    unsigned int requested_team_size,            /* requested number of threads in team */
    int flags,                                   /* flags */
    const void *codeptr_ra                       /* return address of runtime call      */
) {
    if (!enabled) { return; }
    APEX_UNUSED(encountering_task_data);
    APEX_UNUSED(encountering_task_frame);
    APEX_UNUSED(requested_team_size);
    APEX_UNUSED(flags);
    char regionIDstr[128] = {0};
    snprintf(regionIDstr, 128, "OpenMP Parallel Region: UNRESOLVED ADDR %p", codeptr_ra);
    apex_ompt_start(regionIDstr, parallel_data, encountering_task_data, true, codeptr_ra, true);
    DEBUG_PRINT("%" PRId64 ": Parallel Region Begin parent: %p, apex_parent: %p, region: %p, apex_region: %p, %s\n", apex_threadid, (void*)encountering_task_data, encountering_task_data->ptr, (void*)parallel_data, parallel_data->ptr, regionIDstr);
}

/* Event #4, parallel region end */
static void apex_parallel_region_end (
    ompt_data_t *parallel_data,           /* data of parallel region             */
    ompt_data_t *encountering_task_data,  /* data of encountering task           */
    int flags,                            /* flags              */
    const void *codeptr_ra                /* return address of runtime call      */
) {
    if (!enabled) { return; }
    APEX_UNUSED(encountering_task_data);
    APEX_UNUSED(flags);
    APEX_UNUSED(codeptr_ra);
    DEBUG_PRINT("%" PRId64 ": Parallel Region End parent: %p, apex_parent: %p, region: %p, apex_region: %p\n", apex_threadid, (void*)encountering_task_data, encountering_task_data->ptr, (void*)parallel_data, parallel_data->ptr);
    apex_ompt_stop(parallel_data);
}

/* Event #5, task create */
extern "C" void apex_task_create (
    ompt_data_t *encountering_task_data,        /* data of parent task            */
    const ompt_frame_t *encountering_task_frame, /* frame data for parent task     */
    ompt_data_t *new_task_data,                 /* data of created task           */
    int type,                                   /* flags */
    int has_dependences,                        /* created task has dependences   */
    const void *codeptr_ra                      /* return address of runtime call */
) {
    if (!enabled) { return; }
    APEX_UNUSED(encountering_task_frame);
    APEX_UNUSED(has_dependences);
    char * type_str;
    static const char * initial_str = "OpenMP Initial Task";
    static const char * implicit_str = "OpenMP Implicit Task";
    static const char * explicit_str = "OpenMP Explicit Task";
    static const char * target_str = "OpenMP Target Task";
    static const char * undeferred_str = "OpenMP Undeferred Task";
    static const char * untied_str = "OpenMP Untied Task";
    static const char * final_str = "OpenMP Final Task";
    static const char * mergable_str = "OpenMP Mergable Task";
    static const char * merged_str = "OpenMP Merged Task";
    switch ((ompt_task_flag_t)(type)) {
        case ompt_task_initial:
            type_str = const_cast<char*>(initial_str);
            break;
        case ompt_task_implicit:
            type_str = const_cast<char*>(implicit_str);
            break;
        case ompt_task_explicit:
            type_str = const_cast<char*>(explicit_str);
            break;
        case ompt_task_target:
            type_str = const_cast<char*>(target_str);
            break;
        case ompt_task_undeferred:
            type_str = const_cast<char*>(undeferred_str);
            break;
        case ompt_task_untied:
            type_str = const_cast<char*>(untied_str);
            break;
        case ompt_task_final:
            type_str = const_cast<char*>(final_str);
            break;
        case ompt_task_mergeable:
            type_str = const_cast<char*>(mergable_str);
            break;
        case ompt_task_merged:
        default:
            type_str = const_cast<char*>(merged_str);
    }
    DEBUG_PRINT("%" PRId64 ": %s Task Create parent: %p, child: %p\n", apex_threadid, type_str, (void*)encountering_task_data, (void*)new_task_data);

    if (codeptr_ra != nullptr) {
        char regionIDstr[128] = {0};
        snprintf(regionIDstr, 128, "%s: UNRESOLVED ADDR %p", type_str, codeptr_ra);
        apex_ompt_start(regionIDstr, new_task_data, encountering_task_data,
        false, codeptr_ra);
    } else {
        apex_ompt_start(type_str, new_task_data, encountering_task_data,
        false);
    }
}

/* Event #6, task schedule */
extern "C" void apex_task_schedule(
    ompt_data_t *prior_task_data,         /* data of prior task   */
    ompt_task_status_t prior_task_status, /* status of prior task */
    ompt_data_t *next_task_data           /* data of next task    */
    ) {
    if (!enabled) { return; }
    DEBUG_PRINT("%" PRId64 ": Task Schedule prior: %p, status: %d, next: %p\n", apex_threadid, (void*)prior_task_data, prior_task_status, (void*)next_task_data);
    if (prior_task_data != nullptr) {
        linked_timer* prior = (linked_timer*)(prior_task_data->ptr);
        if (prior != nullptr) {
            switch (prior_task_status) {
                case ompt_task_yield:
                case ompt_task_detach:
                case ompt_task_switch:
                    prior->yield();
                    break;
                case ompt_task_complete:
                case ompt_task_early_fulfill:
                case ompt_task_late_fulfill:
                case ompt_task_cancel:
                default:
                    void* tmp = prior->prev;
                    delete(prior);
                    prior_task_data->ptr = tmp;
            }
        }
    }
    //apex_ompt_start("OpenMP Task", next_task_data, nullptr, true);
    linked_timer* next = (linked_timer*)(next_task_data->ptr);
    //APEX_ASSERT(next);
    if (next != nullptr) {
        next->start();
    }
}

/* Event #7, implicit task */
extern "C" void apex_implicit_task(
    ompt_scope_endpoint_t endpoint, /* endpoint of implicit task       */
    ompt_data_t *parallel_data,     /* data of parallel region         */
    ompt_data_t *task_data,         /* data of implicit task           */
    unsigned int team_size,         /* team size                       */
    unsigned int thread_num,        /* thread number of calling thread */
    int flags
  ) {
    if (!enabled) { return; }
    /* Initial tasks confuse the callpath/taskgraph, so don't process them */
    if (flags == ompt_task_initial) { return; }
    APEX_UNUSED(team_size);
    APEX_UNUSED(thread_num);
    APEX_UNUSED(flags);
    if (endpoint == ompt_scope_begin) {
        std::stringstream ss;
        void * codeptr = nullptr;
        if (flags == ompt_task_initial) {
            ss << "OpenMP Initial Task";
        } else {
            ss << "OpenMP Implicit Task";
        }
        /* If the implicit task is from a parallel region, we want to make
         * this timer unique by adding the address of the parallel region. */
        if (parallel_data != nullptr && parallel_data->ptr != nullptr) {
            linked_timer* parent = (linked_timer*)(parallel_data->ptr);
            if (parent->codeptr != nullptr) {
                ss << ": UNRESOLVED ADDR " << parent->codeptr;
                codeptr = (void*)parent->codeptr;
            }
        }
        std::string tmp{ss.str()};
        apex_ompt_start(tmp.c_str(), task_data, parallel_data, true, codeptr);
        if (flags == ompt_task_initial) {
            the_initial_task = task_data;
        }
    } else {
        apex_ompt_stop(task_data);
        if (flags == ompt_task_initial && task_data == the_initial_task) {
            the_initial_task = nullptr;
        }
    }
    DEBUG_PRINT("%" PRId64 ": Initial/Implicit Task task [%u:%u]: %p, apex: %p, region: %p, region ptr: %p, %d\n",
        apex_threadid, thread_num, team_size, (void*)task_data, task_data->ptr,
        (void*)parallel_data, parallel_data ? parallel_data->ptr : nullptr,
        endpoint);
}

/* These are placeholder functions */

#if 1

/* Event #8, target */
extern "C" void apex_target (
    ompt_target_t kind,
    ompt_scope_endpoint_t endpoint,
    int device_num,
    ompt_data_t *task_data,
    ompt_id_t target_id,
    const void *codeptr_ra
) {
    APEX_UNUSED(kind);
    APEX_UNUSED(device_num);
    APEX_UNUSED(target_id);
    if (!enabled) { return; }
    static std::mutex target_lock;
    static std::unordered_map<ompt_id_t, ompt_data_t*> target_map;
    DEBUG_PRINT("Callback Target:\n"
        "\ttarget_id=%lu kind=%d endpoint=%d device_num=%d code=%p,\n"
        "\ttask_data->value=%" PRId64 ", task_data->ptr=%p\n",
        target_id, kind, endpoint, device_num, codeptr_ra,
        task_data == nullptr ? 0 : task_data->value, task_data == nullptr ? nullptr : task_data->ptr);
    if (endpoint == ompt_scope_begin) {
        if (task_data == nullptr) {
            task_data = new ompt_data_t;
            task_data->value = 0;
            task_data->ptr = nullptr;
        }
        char regionIDstr[128] = {0};
        if (codeptr_ra != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP Target: UNRESOLVED ADDR %p",
            codeptr_ra);
            apex_ompt_start(regionIDstr, task_data, nullptr, true, codeptr_ra);
        } else {
            snprintf(regionIDstr, 128, "OpenMP Target");
            apex_ompt_start(regionIDstr, task_data, nullptr, true);
        }
        {
            std::unique_lock<std::mutex> l(target_lock);
            target_map[target_id] = task_data;
        }
        // save the timer data to make flow events
        std::shared_ptr<apex::task_wrapper> tw = ((linked_timer*)(task_data->ptr))->tw;
        apex::async_event_data as_data(
                tw->prof->get_start_ns(),
                "ControlFlow", target_id,
                apex::thread_instance::get_id(), "OpenMP Target");
        OmptGlobals::insert_data(target_id, as_data);
        // save a copy of the task wrapper
        OmptGlobals::insert_timer(target_id, tw);
        static bool doOnce{OmptGlobals::set_delta_base()};
        APEX_UNUSED(doOnce);
    } else {
        {
            std::unique_lock<std::mutex> l(target_lock);
            if (task_data == nullptr) {
                task_data = target_map[target_id];
            }
            target_map.erase(target_id);
        }
        // stop the timer
        apex_ompt_stop(task_data);
        OmptGlobals::update_data(target_id);
    }
}

/* Event #9, target data */
extern "C" void apex_target_data_op (
    ompt_id_t target_id,
    ompt_id_t host_op_id,
    ompt_target_data_op_t optype,
    void *src_addr,
    int src_device_num,
    void *dest_addr,
    int dest_device_num,
    size_t bytes,
    const void *codeptr_ra
) {
    APEX_UNUSED(target_id);
    APEX_UNUSED(host_op_id);
    APEX_UNUSED(optype);
    APEX_UNUSED(src_addr);
    APEX_UNUSED(src_device_num);
    APEX_UNUSED(dest_addr);
    APEX_UNUSED(dest_device_num);
    APEX_UNUSED(bytes);
    APEX_UNUSED(codeptr_ra);
    static std::unordered_map<void*,size_t> allocations;
    static std::mutex allocation_lock;
    if (!enabled) { return; }
    DEBUG_PRINT("Callback DataOp:\n"
        "\ttarget_id=%lu host_op_id=%lu optype=%d src=%p\n"
        "\tsrc_device_num=%d dest=%p dest_device_num=%d bytes=%lu code=%p\n",
        target_id, host_op_id, optype, src_addr, src_device_num,
        dest_addr, dest_device_num, bytes, codeptr_ra);
    // get the address and save the bytes transferred
    switch (optype) {
        case ompt_target_data_alloc: {
            apex::sample_value("GPU: OpenMP Target Data Alloc",bytes);
            std::unique_lock<std::mutex> l(allocation_lock);
            allocations[src_addr] = bytes;
            //apex::rsmi::monitor::instance().explicitMemCheck();
            break;
        }
        case ompt_target_data_transfer_to_device: {
            apex::sample_value("GPU: OpenMP Target Data Transfer to Device",bytes);
            std::unique_lock<std::mutex> l(allocation_lock);
            allocations[dest_addr] = bytes;
            break;
        }
        case ompt_target_data_transfer_from_device: {
            apex::sample_value("GPU: OpenMP Target Data Transfer from Device",bytes);
            std::unique_lock<std::mutex> l(allocation_lock);
            allocations[dest_addr] = bytes;
            break;
        }
        case ompt_target_data_delete: {
            size_t mybytes;
            {
                std::unique_lock<std::mutex> l(allocation_lock);
                mybytes = allocations[src_addr];
                allocations.erase(src_addr);
            }
            apex::sample_value("GPU: OpenMP Target Data Delete",mybytes);
            //apex::rsmi::monitor::instance().explicitMemCheck();
            break;
        }
        case ompt_target_data_associate: {
            apex::sample_value("GPU: OpenMP Target Data Associate",bytes);
            break;
        }
        case ompt_target_data_disassociate: {
            apex::sample_value("GPU: OpenMP Target Data Disassociate",bytes);
            break;
        }
#if defined(APEX_HAVE_OMPT_5_1)
        case ompt_target_data_alloc_async: {
            apex::sample_value("GPU: OpenMP Target Data Alloc Async",bytes);
            std::unique_lock<std::mutex> l(allocation_lock);
            allocations[src_addr] = bytes;
            //apex::rsmi::monitor::instance().explicitMemCheck();
            break;
        }
        case ompt_target_data_transfer_to_device_async: {
            apex::sample_value("GPU: OpenMP Target Data Transfer to Device Async",bytes);
            std::unique_lock<std::mutex> l(allocation_lock);
            allocations[dest_addr] = bytes;
            break;
        }
        case ompt_target_data_transfer_from_device_async: {
            apex::sample_value("GPU: OpenMP Target Data Transfer from Device Async",bytes);
            std::unique_lock<std::mutex> l(allocation_lock);
            allocations[dest_addr] = bytes;
            break;
        }
        case ompt_target_data_delete_async: {
            size_t mybytes;
            {
                std::unique_lock<std::mutex> l(allocation_lock);
                mybytes = allocations[src_addr];
                allocations.erase(src_addr);
            }
            apex::sample_value("GPU: OpenMP Target Data Delete Async",mybytes);
            //apex::rsmi::monitor::instance().explicitMemCheck();
            break;
        }
#endif
        default:
            apex::sample_value("GPU: OpenMP Target Data other",bytes);
            break;
    }
}

/* Event #10, target submit */
extern "C" void apex_target_submit (
    ompt_id_t target_id,
    ompt_id_t host_op_id,
    unsigned int requested_num_teams
) {
    APEX_UNUSED(target_id);
    APEX_UNUSED(host_op_id);
    APEX_UNUSED(requested_num_teams);
    if (!enabled) { return; }
    DEBUG_PRINT("Callback Submit:\n"
        "\ttarget_id=%lu host_op_id=%lu req_num_teams=%d\n",
        target_id, host_op_id, requested_num_teams);
}

/* Event #11, tool control */
extern "C" void apex_control(
    uint64_t command,      /* command of control call             */
    uint64_t modifier,     /* modifier of control call            */
    void *arg,             /* argument of control call            */
    const void *codeptr_ra /* return address of runtime call      */
    ) {
    APEX_UNUSED(command);
    APEX_UNUSED(modifier);
    APEX_UNUSED(arg);
    APEX_UNUSED(codeptr_ra);
    if (!enabled) { return; }
    DEBUG_PRINT("%s\n", __APEX_FUNCTION__);
}

// forward declare some functions
static ompt_set_result_t apex_set_trace_ompt();
static int apex_ompt_start_trace();

/* Event #12, device initialize */
extern "C" void apex_device_initialize (
    int device_num,
    const char *type,
    ompt_device_t *device,
    ompt_function_lookup_t lookup,
    const char *documentation
) {
    APEX_UNUSED(device_num);
    APEX_UNUSED(type);
    APEX_UNUSED(device);
    APEX_UNUSED(documentation);
    if (!enabled) { return; }
    DEBUG_PRINT("%s\n", __APEX_FUNCTION__);
    DEBUG_PRINT("Init: device_num=%d type=%s device=%p lookup=%p doc=%p\n",
        device_num, type, device, (void*)lookup, (void*)documentation);
    if (!lookup) {
        DEBUG_PRINT("Trace collection disabled on device %d\n", device_num);
        return;
    }

    ompt_get_device_time = (ompt_get_device_time_t) lookup("ompt_get_device_time");
    ompt_translate_time = (ompt_translate_time_t) lookup("ompt_translate_time");
    ompt_set_trace_ompt = (ompt_set_trace_ompt_t) lookup("ompt_set_trace_ompt");
    ompt_start_trace = (ompt_start_trace_t) lookup("ompt_start_trace");
    ompt_flush_trace = (ompt_flush_trace_t) lookup("ompt_flush_trace");
    ompt_stop_trace = (ompt_stop_trace_t) lookup("ompt_stop_trace");
    ompt_get_record_ompt = (ompt_get_record_ompt_t) lookup("ompt_get_record_ompt");
    ompt_advance_buffer_cursor = (ompt_advance_buffer_cursor_t) lookup("ompt_advance_buffer_cursor");

    apex_set_trace_ompt();

    // In many scenarios, this will be a good place to start the
    // trace. If start_trace is called from the main program before this
    // callback is dispatched, the start_trace handle will be null. This
    // is because this device_init callback is invoked during the first
    // target construct implementation.

    apex_ompt_start_trace();
}

static uint64_t apex_ompt_translate_time(int index, ompt_device_time_t time) {
    ompt_device_t* device = OmptGlobals::get_device(index);
    if (ompt_translate_time == nullptr) {
        /* runtime doesn't provide a time conversion function... */
        if (ompt_get_device_time == nullptr) {
            /* oh no. we can't get a timestamp. */
            /* take an offset from the first host-side GPU call */
            static bool doOnce{OmptGlobals::set_delta(time)};
            APEX_UNUSED(doOnce);
            return (uint64_t)(time + OmptGlobals::delta());
        }
        static std::map<int, int64_t> deltas;
        if (deltas.count(index) == 0) {
            // get a timestamp
            auto gpu = ompt_get_device_time(device);
            // get a timestamp
            auto cpu = apex::profiler::now_ns();
            // get a delta
            int64_t delta = gpu - cpu;
            deltas.insert(std::pair<int, int64_t>(index,delta));
        }
        return (uint64_t)(time + deltas[index]);
    }
    // the time gets returned as seconds, so convert to nanoseconds
    return (uint64_t)(ompt_translate_time(device, time) * 1.0e9);
}


static int apex_ompt_flush_trace();
static int apex_ompt_stop_trace();

/* Event #13, device finalize */
extern "C" void apex_device_finalize (
    uint64_t device_num
) {
    APEX_UNUSED(device_num);
    if (!enabled) { return; }
    DEBUG_PRINT("%s\n", __APEX_FUNCTION__);
    apex_ompt_flush_trace();
    apex_ompt_stop_trace();
}

/* Event #14, device load */
extern "C" void apex_device_load (
    uint64_t device_num,
    const char * filename,
    int64_t offset_in_file,
    void * vma_in_file,
    size_t bytes,
    void * host_addr,
    void * device_addr,
    uint64_t module_id
) {
    APEX_UNUSED(device_num);
    APEX_UNUSED(filename);
    APEX_UNUSED(offset_in_file);
    APEX_UNUSED(vma_in_file);
    APEX_UNUSED(bytes);
    APEX_UNUSED(host_addr);
    APEX_UNUSED(device_addr);
    APEX_UNUSED(module_id);
    if (!enabled) { return; }
    DEBUG_PRINT("%s\n", __APEX_FUNCTION__);
}

/* Event #15, device load */
extern "C" void apex_device_unload (
    uint64_t device_num,
    uint64_t module_id
) {
    APEX_UNUSED(device_num);
    APEX_UNUSED(module_id);
    if (!enabled) { return; }
    DEBUG_PRINT("%s\n", __APEX_FUNCTION__);
}

/* Event #22, target map */
extern "C" void apex_target_map (
    ompt_id_t target_id,
    unsigned int nitems,
    void **host_addr,
    void **device_addr,
    size_t *bytes,
    unsigned int *mapping_flags,
    const void *codeptr_ra
) {
    APEX_UNUSED(target_id);
    APEX_UNUSED(nitems);
    APEX_UNUSED(host_addr);
    APEX_UNUSED(device_addr);
    APEX_UNUSED(bytes);
    APEX_UNUSED(mapping_flags);
    APEX_UNUSED(codeptr_ra);
    if (!enabled) { return; }
    DEBUG_PRINT("%s\n", __APEX_FUNCTION__);
    // get the direction, and capture the bytes transferred.
}

#endif // placeholder functions

/**********************************************************************/
/* End Mandatory Events */
/**********************************************************************/

/**********************************************************************/
/* Optional events */
/**********************************************************************/

/* Event #16, sync region wait       */
extern "C" void apex_sync_region_wait (
    ompt_sync_region_t kind,        /* kind of sync region            */
    ompt_scope_endpoint_t endpoint, /* endpoint of sync region        */
    ompt_data_t *parallel_data,     /* data of parallel region        */
    ompt_data_t *task_data,         /* data of task                   */
    const void *codeptr_ra          /* return address of runtime call */
) {
    if (!enabled) { return; }
    char * tmp_str;
    static const char * barrier_str = "Barrier Wait";
    static const char * barrier_i_str = "Implicit Barrier Wait";
    static const char * barrier_e_str = "Explicit Barrier Wait";
    static const char * barrier_imp_str = "Barrier Implementation Wait";
    static const char * task_wait_str = "Task Wait";
    static const char * task_group_str = "Task Group Wait";
    static const char * reduction_str = "Reduction Wait";
    static const char * unknown_str = "Unknown Wait";
    switch (kind) {
        case ompt_sync_region_barrier:
            tmp_str = const_cast<char*>(barrier_str);
            break;
        case ompt_sync_region_barrier_implicit:
            tmp_str = const_cast<char*>(barrier_i_str);
            break;
        case ompt_sync_region_barrier_explicit:
            tmp_str = const_cast<char*>(barrier_e_str);
            break;
        case ompt_sync_region_barrier_implementation:
            tmp_str = const_cast<char*>(barrier_imp_str);
            break;
        case ompt_sync_region_taskwait:
            tmp_str = const_cast<char*>(task_wait_str);
            break;
        case ompt_sync_region_taskgroup:
            tmp_str = const_cast<char*>(task_group_str);
            break;
        case ompt_sync_region_reduction:
            tmp_str = const_cast<char*>(reduction_str);
            break;
#if defined(APEX_HAVE_OMPT_5_1)
        case ompt_sync_region_barrier_implicit_workshare:
            static const char * barrier_implicit_workshare_str = "Barrier Implicit Workshare Wait";
            tmp_str = const_cast<char*>(barrier_implicit_workshare_str);
            break;
        case ompt_sync_region_barrier_implicit_parallel:
            static const char * barrier_implicit_parallel_str = "Barrier Implicit Parallel Wait";
            tmp_str = const_cast<char*>(barrier_implicit_parallel_str);
            break;
        case ompt_sync_region_barrier_teams:
            static const char * barrier_teams_str = "Barrier Teams Wait";
            tmp_str = const_cast<char*>(barrier_teams_str);
            break;
#endif
        default:
            tmp_str = const_cast<char*>(unknown_str);
            break;
    }
    if (endpoint == ompt_scope_begin) {
        void * local_codeptr = (void*)codeptr_ra;
        /* THis code isn't quite correct.  After the end of a parallel
         * region, the threads in the team will sit in an unidentified
         * barrier, waiting for the next parallel region where they
         * are needed.  So... should that barrier be associated with
         * the previous parallel region, or just be anonymous? */

        /* If OpenMP doesn't give us a codeptr, use the one from the
         * parent, if it has one */
        if (codeptr_ra == nullptr &&
            apex::apex_options::ompt_high_overhead_events()) {
            if (parallel_data != nullptr && parallel_data->ptr != nullptr) {
                linked_timer* parent = (linked_timer*)(parallel_data->ptr);
                if (parent->codeptr != nullptr) {
                    local_codeptr = (void*)parent->codeptr;
                }
            }
        }
        char regionIDstr[128] = {0};
        if (local_codeptr != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP %s: UNRESOLVED ADDR %p", tmp_str,
                local_codeptr);
            apex_ompt_start(regionIDstr, task_data, parallel_data, true,
                local_codeptr);
        } else {
            snprintf(regionIDstr, 128, "OpenMP %s", tmp_str);
            apex_ompt_start(regionIDstr, task_data, parallel_data, true);
        }
    } else {
        apex_ompt_stop(task_data);
    }
    DEBUG_PRINT("%" PRId64 ": %s: %p, apex: %p, region: %p, region ptr: %p, %d\n",
        apex_threadid, tmp_str, (void*)task_data, task_data->ptr,
        (void*)parallel_data, parallel_data ? parallel_data->ptr : nullptr,
        endpoint);
}

/* Event #20, task at work begin or end       */
extern "C" void apex_ompt_work (
    ompt_work_t wstype,             /* type of work region            */
    ompt_scope_endpoint_t endpoint, /* endpoint of work region        */
    ompt_data_t *parallel_data,     /* data of parallel region        */
    ompt_data_t *task_data,         /* data of task                   */
    uint64_t count,                 /* quantity of work               */
    const void *codeptr_ra          /* return address of runtime call */
    ) {
    if (!enabled) { return; }
    APEX_UNUSED(count); // unused on end

    char * tmp_str;
    static const char * loop_str = "Loop";
    static const char * sections_str = "Sections";
    static const char * single_executor_str = "Single Executor";
    static const char * single_other_str = "Single Other";
    static const char * workshare_str = "Workshare";
    static const char * distribute_str = "Distribute";
    static const char * taskloop_str = "Taskloop";
    static const char * unknown_str = "Unknown";

    /*
    static const char * iterations_type = "Iterations";
    static const char * collapsed_type = "Iterations (collapsed)";
    static const char * sections_type = "Sections";
    static const char * units_type = "Units of Work";
    static const char * single_type = "Single";
    char * count_type = const_cast<char*>(iterations_type);
    */

    switch(wstype) {
        case ompt_work_loop:
            tmp_str = const_cast<char*>(loop_str);
            break;
        case ompt_work_sections:
            tmp_str = const_cast<char*>(sections_str);
            //count_type = const_cast<char*>(sections_type);
            break;
        case ompt_work_single_executor:
            tmp_str = const_cast<char*>(single_executor_str);
            //count_type = const_cast<char*>(single_type);
            break;
        case ompt_work_single_other:
            tmp_str = const_cast<char*>(single_other_str);
            //count_type = const_cast<char*>(single_type);
            break;
        case ompt_work_workshare:
            tmp_str = const_cast<char*>(workshare_str);
            //count_type = const_cast<char*>(units_type);
            break;
        case ompt_work_distribute:
            tmp_str = const_cast<char*>(distribute_str);
            break;
        case ompt_work_taskloop:
            tmp_str = const_cast<char*>(taskloop_str);
            //count_type = const_cast<char*>(collapsed_type);
            break;
        default:
            tmp_str = const_cast<char*>(unknown_str);
            break;
    }
    if (endpoint == ompt_scope_begin) {
        char regionIDstr[128] = {0};
        DEBUG_PRINT("%" PRId64 ": %s Begin task: %p, region: %p\n", apex_threadid,
        tmp_str, (void*)task_data, (void*)parallel_data);
        if (codeptr_ra != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP Work %s: UNRESOLVED ADDR %p", tmp_str,
            codeptr_ra);
            apex_ompt_start(regionIDstr, task_data, parallel_data, true, codeptr_ra);
        } else {
            snprintf(regionIDstr, 128, "OpenMP Work %s", tmp_str);
            apex_ompt_start(regionIDstr, task_data, parallel_data, true);
        }
        /*
        if (apex::apex_options::ompt_high_overhead_events()) {
            std::stringstream ss;
            ss << count_type << ": " << regionIDstr;
            std::string tmp{ss.str()};
            apex::sample_value(tmp, count);
        }
        */
    } else {
        DEBUG_PRINT("%" PRId64 ": %s End task: %p, region: %p\n", apex_threadid, tmp_str,
        (void*)task_data, (void*)parallel_data);
        apex_ompt_stop(task_data);
    }
}

/* Event #21, task at master begin or end       */
extern "C" void apex_ompt_master (
    ompt_scope_endpoint_t endpoint, /* endpoint of master region           */
    ompt_data_t *parallel_data,     /* data of parallel region             */
    ompt_data_t *task_data,         /* data of task                        */
    const void *codeptr_ra          /* return address of runtime call      */
) {
    if (!enabled) { return; }
    if (endpoint == ompt_scope_begin) {
        if (codeptr_ra != nullptr) {
            char regionIDstr[128] = {0};
            snprintf(regionIDstr, 128, "OpenMP Master: UNRESOLVED ADDR %p",
            codeptr_ra);
            apex_ompt_start(regionIDstr, task_data, parallel_data, true, codeptr_ra);
        } else {
            apex_ompt_start("OpenMP Master", task_data, parallel_data, true);
        }
    } else {
        apex_ompt_stop(task_data);
    }
}

/* Event #23, sync region begin or end */
extern "C" void apex_ompt_sync_region (
    ompt_sync_region_t kind,        /* kind of sync region                 */
    ompt_scope_endpoint_t endpoint, /* endpoint of sync region             */
    ompt_data_t *parallel_data,     /* data of parallel region             */
    ompt_data_t *task_data,         /* data of task                        */
    const void *codeptr_ra          /* return address of runtime call      */
) {
    char * tmp_str;
    static const char * barrier_str = "Barrier";
    static const char * barrier_i_str = "Implicit Barrier";
    static const char * barrier_e_str = "Explicit Barrier";
    static const char * barrier_imp_str = "Barrier Implementation";
    static const char * task_str = "Task";
    static const char * task_group_str = "Task Group";
    static const char * reduction_str = "Reduction";
    static const char * unknown_str = "Unknown";
    switch (kind) {
        case ompt_sync_region_barrier:
            tmp_str = const_cast<char*>(barrier_str);
            break;
        case ompt_sync_region_barrier_implicit:
            tmp_str = const_cast<char*>(barrier_i_str);
            break;
        case ompt_sync_region_barrier_explicit:
            tmp_str = const_cast<char*>(barrier_e_str);
            break;
        case ompt_sync_region_barrier_implementation:
            tmp_str = const_cast<char*>(barrier_imp_str);
            break;
        case ompt_sync_region_taskwait:
            tmp_str = const_cast<char*>(task_str);
            break;
        case ompt_sync_region_taskgroup:
            tmp_str = const_cast<char*>(task_group_str);
            break;
        case ompt_sync_region_reduction:
            tmp_str = const_cast<char*>(reduction_str);
            break;
#if defined(APEX_HAVE_OMPT_5_1)
        case ompt_sync_region_barrier_implicit_workshare:
            static const char * barrier_implicit_workshare_str = "Barrier Implicit Workshare";
            tmp_str = const_cast<char*>(barrier_implicit_workshare_str);
            break;
        case ompt_sync_region_barrier_implicit_parallel:
            static const char * barrier_implicit_parallel_str = "Barrier Implicit Parallel";
            tmp_str = const_cast<char*>(barrier_implicit_parallel_str);
            break;
        case ompt_sync_region_barrier_teams:
            static const char * barrier_teams_str = "Barrier Teams";
            tmp_str = const_cast<char*>(barrier_teams_str);
            break;
#endif
        default:
            tmp_str = const_cast<char*>(unknown_str);
            break;
    }
    if (endpoint == ompt_scope_begin) {
        void * local_codeptr = (void*)codeptr_ra;
        /* THis code isn't quite correct.  After the end of a parallel
         * region, the threads in the team will sit in an unidentified
         * barrier, waiting for the next parallel region where they
         * are needed.  So... should that barrier be associated with
         * the previous parallel region, or just be anonymous? */
#if 1
        /* If OpenMP doesn't give us a codeptr, use the one from the
         * parent, if it has one */
        if (codeptr_ra == nullptr &&
            apex::apex_options::ompt_high_overhead_events()) {
            if (parallel_data != nullptr && parallel_data->ptr != nullptr) {
                linked_timer* parent = (linked_timer*)(parallel_data->ptr);
                if (parent->codeptr != nullptr) {
                    local_codeptr = (void*)parent->codeptr;
                }
            }
        }
#endif
        char regionIDstr[128] = {0};
        if (local_codeptr != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP %s: UNRESOLVED ADDR %p", tmp_str,
                local_codeptr);
            apex_ompt_start(regionIDstr, task_data, parallel_data, true,
                local_codeptr);
        } else {
            snprintf(regionIDstr, 128, "OpenMP %s", tmp_str);
            apex_ompt_start(regionIDstr, task_data, parallel_data, true);
        }
    } else {
        apex_ompt_stop(task_data);
    }
}

/* Event #29, flush event */
extern "C" void apex_ompt_flush (
    ompt_data_t *thread_data, /* data of thread                      */
    const void *codeptr_ra    /* return address of runtime call      */
) {
    if (!enabled) { return; }
    APEX_UNUSED(thread_data);
    if (codeptr_ra != nullptr) {
        char regionIDstr[128] = {0};
        snprintf(regionIDstr, 128, "OpenMP Flush: UNRESOLVED ADDR %p", codeptr_ra);
        apex::sample_value(regionIDstr, 1);
    } else {
        apex::sample_value(std::string("OpenMP Flush"),1);
    }
}

/* Event #30, cancel event */
extern "C" void apex_ompt_cancel (
    ompt_data_t *task_data,   /* data of task                        */
    int flags,                /* cancel flags                        */
    const void *codeptr_ra    /* return address of runtime call      */
) {
    if (!enabled) { return; }
    char regionIDstr[128] = {0};
    if (flags & ompt_cancel_parallel) {
        if (codeptr_ra != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP Cancel Parallel: UNRESOLVED ADDR %p",
            codeptr_ra);
            apex::sample_value(std::string(regionIDstr),1);
        } else {
            apex::sample_value(std::string("OpenMP Cancel Parallel"),1);
        }
    }
    if (flags & ompt_cancel_sections) {
        if (codeptr_ra != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP Cancel Sections: UNRESOLVED ADDR %p",
            codeptr_ra);
            apex::sample_value(std::string(regionIDstr),1);
        } else {
            apex::sample_value(std::string("OpenMP Cancel Sections"),1);
        }
    }
    if (flags & ompt_cancel_loop) {
        if (codeptr_ra != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP Cancel Do: UNRESOLVED ADDR %p",
            codeptr_ra);
            apex::sample_value(std::string(regionIDstr),1);
        } else {
            apex::sample_value(std::string("OpenMP Cancel Do"),1);
        }
    }
    if (flags & ompt_cancel_taskgroup) {
        if (codeptr_ra != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP Cancel Taskgroup: UNRESOLVED ADDR %p",
            codeptr_ra);
            apex::sample_value(std::string(regionIDstr),1);
        } else {
            apex::sample_value(std::string("OpenMP Cancel Taskgroup"),1);
        }
    }
    if (flags & ompt_cancel_activated) {
        if (codeptr_ra != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP Cancel Activated: UNRESOLVED ADDR %p",
            codeptr_ra);
            apex::sample_value(std::string(regionIDstr),1);
        } else {
            apex::sample_value(std::string("OpenMP Cancel Activated"),1);
        }
    }
    if (flags & ompt_cancel_detected) {
        if (codeptr_ra != nullptr) {
            snprintf(regionIDstr, 128, "OpenMP Cancel Detected: UNRESOLVED ADDR %p",
            codeptr_ra);
            apex::sample_value(std::string(regionIDstr),1);
        } else {
            apex::sample_value(std::string("OpenMP Cancel Detected"),1);
        }
    }
    if (flags & ompt_cancel_discarded_task) {
        if (codeptr_ra != nullptr) {
            snprintf(regionIDstr, 128,
            "OpenMP Cancel Discarded Task: UNRESOLVED ADDR %p", codeptr_ra);
            apex::sample_value(std::string(regionIDstr),1);
        } else {
            apex::sample_value(std::string("OpenMP Cancel Discarded Task"),1);
        }
    }
    apex_ompt_stop(task_data);
}

/* Event #30, cancel event */
extern "C" void apex_ompt_idle (
    ompt_scope_endpoint_t endpoint /* endpoint of idle time               */
) {
    if (!enabled) { return; }
    static APEX_NATIVE_TLS apex::profiler* p = nullptr;
    if (endpoint == ompt_scope_begin) {
        p = apex::start("OpenMP Idle");
    } else {
        apex::stop(p);
    }
}

/**********************************************************************/
/* End Optional events */
/**********************************************************************/

/**********************************************************************/
/* Start Asynchronous handling callbacks */
/**********************************************************************/

// Trace record callbacks
// Allocation routine
static void on_ompt_callback_buffer_request (
  int device_num,
  ompt_buffer_t **buffer,
  size_t *bytes
) {
    APEX_UNUSED(device_num);
    APEX_UNUSED(buffer);
    APEX_UNUSED(bytes);
  *bytes = apex_ompt_buffer_request_size;
  *buffer = malloc(*bytes);
  //printf("Allocated %lu bytes at %p in buffer request callback\n", *bytes, *buffer);
}

// This function is called by an OpenMP runtime helper thread for
// returning trace records from a buffer.
// Note: This callback must handle a null begin cursor. Currently,
// ompt_get_record_ompt, print_record_ompt, and
// ompt_advance_buffer_cursor handle a null cursor.
static void on_ompt_callback_buffer_complete (
  int device_num,
  ompt_buffer_t *buffer,
  size_t bytes, /* bytes returned in this callback */
  ompt_buffer_cursor_t begin,
  int buffer_owned
) {
    APEX_UNUSED(device_num);
  DEBUG_PRINT("Executing buffer complete callback: %d %p %lu %p %d\n",
     device_num, buffer, bytes, (void*)begin, buffer_owned);

  int status = 1;
  ompt_buffer_cursor_t current = begin;
  while (status) {
    ompt_record_ompt_t *rec = ompt_get_record_ompt(buffer, current);
    print_record_ompt(rec);
    status = ompt_advance_buffer_cursor(NULL, /* TODO device */
                    buffer,
                    bytes,
                    current,
                    &current);
  }
  if (buffer_owned) delete_buffer_ompt(buffer);
}

// Utility routine to enable the desired tracing modes
static ompt_set_result_t apex_set_trace_ompt() {
  if (!ompt_set_trace_ompt) return ompt_set_error;

  ompt_set_trace_ompt(0, 1, ompt_callback_target);
  ompt_set_trace_ompt(0, 1, ompt_callback_target_data_op);
  ompt_set_trace_ompt(0, 1, ompt_callback_target_submit);

  return ompt_set_always;
}

static int apex_ompt_start_trace() {
  if (!ompt_start_trace) return 0;
  return ompt_start_trace(0, &on_ompt_callback_buffer_request,
              &on_ompt_callback_buffer_complete);
}

static int apex_ompt_flush_trace() {
  if (!ompt_flush_trace) return 0;
  return ompt_flush_trace(0);
}

static int apex_ompt_stop_trace() {
  if (!ompt_stop_trace) return 0;
  return ompt_stop_trace(0);
}

/**********************************************************************/
/* End Asynchronous handling callbacks */
/**********************************************************************/

// This function is for checking that the function registration worked.
int apex_ompt_register(ompt_callbacks_t e, ompt_callback_t c ,
    const char * name) {
    APEX_UNUSED(name);
  DEBUG_PRINT("Registering OMPT callback %s...",name); fflush(stderr);
  ompt_set_result_t rc = ompt_set_callback(e, c);
  switch (rc) {
    case ompt_set_error:
        DEBUG_PRINT("\n\tFailed to register OMPT callback %s!\n",name);
        fflush(stderr);
        break;
    case ompt_set_never:
        DEBUG_PRINT("\n\tOMPT callback %s never supported by this runtime.\n",name);
        fflush(stderr);
        break;
    case ompt_set_impossible:
        DEBUG_PRINT("\n\tOMPT callback %s impossible from this runtime.\n",name);
        fflush(stderr);
        break;
    case ompt_set_sometimes:
        DEBUG_PRINT("\n\tOMPT callback %s sometimes supported by this runtime.\n",name);
        fflush(stderr);
        break;
    case ompt_set_sometimes_paired:
        DEBUG_PRINT("\n\tOMPT callback %s sometimes paired by this runtime.\n",name);
        fflush(stderr);
        break;
    case ompt_set_always:
    default:
        DEBUG_PRINT("success.\n");
  }
  return 0;
}

extern "C" {

int ompt_initialize(ompt_function_lookup_t lookup, int initial_device_num,
    ompt_data_t* tool_data) {
    APEX_UNUSED(initial_device_num);
    APEX_UNUSED(tool_data);
    if (!apex::apex_options::use_ompt()) { return 0; }
    if (enabled) { printf("Re-initializing OMPT!\n"); return 1; }
    {
        std::unique_lock<std::mutex> l(apex_apex_threadid_mutex);
        apex_threadid = apex_numthreads++;
    }
    DEBUG_PRINT("Getting OMPT functions..."); fflush(stderr);
    //ompt_function_lookup = lookup;
    ompt_finalize_tool = (ompt_finalize_tool_t)
        lookup("ompt_finalize_tool");
    ompt_set_callback = (ompt_set_callback_t)
        lookup("ompt_set_callback");
        /*
    ompt_get_callback = (ompt_get_callback_t)
        lookup("ompt_get_callback");
    ompt_get_task_info = (ompt_get_task_info_t)
        lookup("ompt_get_task_info");
    ompt_get_task_memory = (ompt_get_task_memory_t)
        lookup("ompt_get_task_memory");
    ompt_get_thread_data = (ompt_get_thread_data_t)
        lookup("ompt_get_thread_data");
    ompt_get_parallel_info = (ompt_get_parallel_info_t)
        lookup("ompt_get_parallel_info");
    ompt_get_unique_id = (ompt_get_unique_id_t)
        lookup("ompt_get_unique_id");
    ompt_get_num_places = (ompt_get_num_places_t)
        lookup("ompt_get_num_places");
    ompt_get_num_devices = (ompt_get_num_devices_t)
        lookup("ompt_get_num_devices");
    ompt_get_num_procs = (ompt_get_num_procs_t)
        lookup("ompt_get_num_procs");
    ompt_get_place_proc_ids = (ompt_get_place_proc_ids_t)
        lookup("ompt_get_place_proc_ids");
    ompt_get_place_num = (ompt_get_place_num_t)
        lookup("ompt_get_place_num");
    ompt_get_partition_place_nums = (ompt_get_partition_place_nums_t)
        lookup("ompt_get_partition_place_nums");
    ompt_get_proc_id = (ompt_get_proc_id_t)
        lookup("ompt_get_proc_id");
    ompt_get_target_info = (ompt_get_target_info_t)
        lookup("ompt_get_target_info");
    ompt_enumerate_states = (ompt_enumerate_states_t)
        lookup("ompt_enumerate_states");
    ompt_enumerate_mutex_impls = (ompt_enumerate_mutex_impls_t)
        lookup("ompt_enumerate_mutex_impls");
        */

    DEBUG_PRINT("success.\n");

    apex::init("OpenMP Program",0,1);
    DEBUG_PRINT("Registering OMPT events...\n"); fflush(stderr);

    /* Mandatory events */

    // Event 1: thread begin
    apex_ompt_register(ompt_callback_thread_begin,
        (ompt_callback_t)&apex_thread_begin, "thread_begin");
    // Event 2: thread end
    apex_ompt_register(ompt_callback_thread_end,
        (ompt_callback_t)&apex_thread_end, "thread_end");
    // Event 3: parallel begin
    apex_ompt_register(ompt_callback_parallel_begin,
        (ompt_callback_t)&apex_parallel_region_begin, "parallel_begin");
    // Event 4: parallel end
    apex_ompt_register(ompt_callback_parallel_end,
        (ompt_callback_t)&apex_parallel_region_end, "parallel_end");
    if (apex::apex_options::ompt_high_overhead_events()) {
        // Event 5: task create
        apex_ompt_register(ompt_callback_task_create,
            (ompt_callback_t)&apex_task_create, "task_create");
        // Event 6: task schedule (start/stop)
        apex_ompt_register(ompt_callback_task_schedule,
            (ompt_callback_t)&apex_task_schedule, "task_schedule");
        // Event 7: implicit task (start/stop)
        apex_ompt_register(ompt_callback_implicit_task,
            (ompt_callback_t)&apex_implicit_task, "implicit_task");
    }

 #if 1
    // Event 8: target
    apex_ompt_register(ompt_callback_target,
        (ompt_callback_t)&apex_target, "target");
    // Event 9: target data operation
    apex_ompt_register(ompt_callback_target_data_op,
        (ompt_callback_t)&apex_target_data_op, "target_data_operation");
    // Event 10: target submit
    apex_ompt_register(ompt_callback_target_submit,
        (ompt_callback_t)&apex_target_submit, "target_submit");
    // Event 11: control tool
    apex_ompt_register(ompt_callback_control_tool,
        (ompt_callback_t)&apex_control, "event_control");
    // Event 12: device initialize
    apex_ompt_register(ompt_callback_device_initialize,
        (ompt_callback_t)&apex_device_initialize, "device_initialize");
    // Event 13: device finalize
    apex_ompt_register(ompt_callback_device_finalize,
        (ompt_callback_t)&apex_device_finalize, "device_finalize");
    // Event 14: device load
    apex_ompt_register(ompt_callback_device_load,
        (ompt_callback_t)&apex_device_load, "device_load");
    // Event 15: device unload
    apex_ompt_register(ompt_callback_device_unload,
        (ompt_callback_t)&apex_device_unload, "device_unload");
#endif

    /* optional events */

    if (!apex::apex_options::ompt_required_events_only()) {
        // Event 20: task at work begin or end
        apex_ompt_register(ompt_callback_work,
            (ompt_callback_t)&apex_ompt_work, "work");
        /* Event 21: task at master begin or end     */
        apex_ompt_register(ompt_callback_master,
            (ompt_callback_t)&apex_ompt_master, "master");
#if 1
        /* Event 22: target map                      */
        apex_ompt_register(ompt_callback_target_map,
            (ompt_callback_t)&apex_target_map, "target_map");
#endif
        /* Event 29: after executing flush           */
        apex_ompt_register(ompt_callback_flush,
            (ompt_callback_t)&apex_ompt_flush, "flush");
        /* Event 30: cancel innermost binding region */
        apex_ompt_register(ompt_callback_cancel,
            (ompt_callback_t)&apex_ompt_cancel, "cancel");

        if (apex::apex_options::ompt_high_overhead_events()) {
            // Event 16: sync region wait begin or end
            apex_ompt_register(ompt_callback_sync_region_wait,
                (ompt_callback_t)&apex_sync_region_wait, "sync_region_wait");
#if 0
            // Event 17: mutex released
            apex_ompt_register(ompt_callback_mutex_released,
                (ompt_callback_t)&apex_mutex_released, "mutex_released");
            // Event 18: report task dependences
            apex_ompt_register(ompt_callback_report_task_dependences,
                (ompt_callback_t)&apex_report_task_dependences,
                "mutex_report_task_dependences");
            // Event 19: report task dependence
            apex_ompt_register(ompt_callback_report_task_dependence,
                (ompt_callback_t)&apex_report_task_dependence,
                "mutex_report_task_dependence");
#endif
            /* Event 23: sync region begin or end        */
            apex_ompt_register(ompt_callback_sync_region,
                (ompt_callback_t)&apex_ompt_sync_region, "sync_region");
            /* Event 31: begin or end idle state         */
//            apex_ompt_register(ompt_callback_idle,
//                (ompt_callback_t)&apex_ompt_idle, "idle");
#if 0
            /* Event 24: lock init                       */
            /* Event 25: lock destroy                    */
            /* Event 26: mutex acquire                   */
            apex_ompt_register(ompt_callback_mutex_acquire,
                (ompt_callback_t)&apex_mutex_acquire, "mutex_acquire");
            /* Event 27: mutex acquired                  */
            apex_ompt_register(ompt_callback_mutex_acquired,
                (ompt_callback_t)&apex_mutex_acquired, "mutex_acquired");
            /* Event 28: nest lock                       */
#endif
        }

    }
    enabled = true;

    DEBUG_PRINT("done.\n"); fflush(stderr);
    return 1;
}

void ompt_finalize(ompt_data_t* tool_data)
{
    APEX_UNUSED(tool_data);
    if (!apex::apex_options::use_ompt()) { return; }
    apex_ompt_flush_trace();
    DEBUG_PRINT("OpenMP runtime is shutting down...\n");
    if (the_initial_task != nullptr) {
        apex_ompt_stop(the_initial_task);
        the_initial_task = nullptr;
    }
    enabled = false;
    apex::finalize();
}

/* According to the OpenMP 5.0 specification, this function needs to be
 * defined in the application address space.  The runtime will see it,
 * and run it. */
ompt_start_tool_result_t * ompt_start_tool(
    unsigned int omp_version, const char *runtime_version) {
    APEX_UNUSED(runtime_version); // in case we aren't printing debug messages
    if (apex::apex_options::use_verbose()) {
        DEBUG_PRINT("APEX: OMPT Tool Start, version %d, '%s'\n",
            omp_version, runtime_version);
#if defined(_OPENMP)
        if (_OPENMP != omp_version) {
        DEBUG_PRINT("APEX: WARNING! %d != %d (OpenMP Version used to compile APEX)\n",
            omp_version, _OPENMP);
        }
#else
        APEX_UNUSED(omp_version); // in case we aren't printing debug messages
#endif
    }
    static ompt_start_tool_result_t result;
    result.initialize = &ompt_initialize;
    result.finalize = &ompt_finalize;
    result.tool_data.value = 0L;
    result.tool_data.ptr = nullptr;
    return &result;
}

/* This function is used by APEX to tell the OpenMP runtime to stop sending
 * OMPT events.  This is when apex::finalize() happens before ompt_finalize().
 * It is called from apex::finalize().
 */
void apex_ompt_force_shutdown(void) {
    DEBUG_PRINT("Forcing shutdown of OpenMP Tools API\n");
    /* The Intel generated code has some odd destructor race conditions, so
     * don't force the runtime to shut down. */
    /* OpenMP might not have been used... */
    if (the_initial_task != nullptr) {
        apex_ompt_stop(the_initial_task);
        the_initial_task = nullptr;
    }
    enabled = false;
    if (ompt_finalize_tool) {
        ompt_finalize_tool();
    }
}

} // extern "C"
