
#include "cperfdmf.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>


int displayTrialData(void) {
  int i;
  int selection;
  char **metrics;
  int numMetrics;

  int ce;

  printf ("loading metrics\n");
  if (CPERFDMF_OK != (ce = cperfdmf_getMetrics(&metrics, &numMetrics))) {
    fprintf (stderr, "error getting metrics: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }

  printf("\nMetrics:\n");
  for (i=0; i<numMetrics; i++) {
    printf("%d) %s\n", i, metrics[i]);
  }
  printf("Select a Metric: ");
  scanf("%d",&selection);
  int metric = selection;


  char **groups;
  int numGroups;

  if (CPERFDMF_OK != (ce = cperfdmf_getGroups(&groups, &numGroups))) {
    fprintf (stderr, "error getting groups: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }



  char **functions;
  int numFunctions;

  if (CPERFDMF_OK != (ce = cperfdmf_getFunctions(&functions, &numFunctions))) {
    fprintf (stderr, "error getting functions: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }

  
  printf("\nFunctions:\n");
  for (i=0; i<numFunctions; i++) {
    printf("%d) %s\n", i, functions[i]);
  }
  printf("Select a Function: ");
  scanf("%d",&selection);


  cperfdmf_profile_t mean, total;

  printf ("Getting Mean Values\n");
  if (CPERFDMF_OK != (ce = cperfdmf_getMeanValues(&mean, selection, metric))) {
    fprintf (stderr, "error getting mean values: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }

  printf ("Getting Total Values\n");
  if (CPERFDMF_OK != (ce = cperfdmf_getTotalValues(&total, selection, metric))) {
    fprintf (stderr, "error getting total values: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }
  

  printf ("\n");
  
  if (mean.numGroups > 0) {
    printf ("Function \"%s\", group \"%s\"\n",functions[selection], groups[mean.groups[0]]);
  } else {
    printf ("Function \"%s\"\n",functions[selection]);
  }

  printf ("N:C:T Exclusive  Inclusive  Exclusive   Inclusive   #Calls   #Subroutines   Incl/Call\n");
  printf ("                              Percent     Percent                    \n");


  printf ("Mean %10.2f %10.2f %10.2f %11.2f %8d %14d %11.2f\n",mean.exclusive / 1000, 
	  mean.inclusive / 1000, mean.exclusivePercent, mean.inclusivePercent,
	  mean.numCalls, mean.numSubroutines, mean.inclusivePerCall / 1000);

  printf ("Total %9.2f %10.2f %10.2f %11.2f %8d %14d %11.2f\n",total.exclusive / 1000, 
	  total.inclusive / 1000, total.exclusivePercent, total.inclusivePercent,
	  total.numCalls, total.numSubroutines, total.inclusivePerCall / 1000);



  int numNodes;
  if (CPERFDMF_OK != (ce = cperfdmf_getNumNodes(&numNodes))) {
    fprintf (stderr, "error getting number of nodes: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }

  int n,c,t;


  int nodesFound = 0;
  for (n=0; nodesFound<numNodes; n++) {

    int exists;
    int ec;

    if (CPERFDMF_OK != (ec = cperfdmf_queryNodeExists(&exists, n))) {
      fprintf (stderr, "Uh oh: %s\n",cperfdmf_errToString(ec));
    }

    if (!exists)
      continue;

    nodesFound++;

    

    int numContexts;
    if (CPERFDMF_OK != (ce = cperfdmf_getNumContexts(&numContexts, n))) {
      fprintf (stderr, "error getting number of contexts: %s\n",cperfdmf_errToString(ce));
      if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
	fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
      }
      return -1;
    }

    int contextsFound = 0;
    
    for (c=0; contextsFound<numContexts; c++) {
      
      if (CPERFDMF_OK != (ec = cperfdmf_queryContextExists(&exists, n, c))) {
	fprintf (stderr, "%s(%d): Error %s\n", __FILE__, __LINE__, cperfdmf_errToString(ec));
      }

      if (!exists)
	continue;
      
      contextsFound++;


      int numThreads;
      if (CPERFDMF_OK != (ce = cperfdmf_getNumThreads(&numThreads, n, c))) {
	fprintf (stderr, "error getting number of threads: %s\n",cperfdmf_errToString(ce));
	if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
	  fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
	}
	return -1;
      }


      int threadsFound=0;
      for (t=0; threadsFound<numThreads; t++) {
	if (CPERFDMF_OK != (ec = cperfdmf_queryThreadExists(&exists, n, c, t))) {
	  fprintf (stderr, "%s(%d): Error %s\n", __FILE__, __LINE__, cperfdmf_errToString(ec));
	}
	
	if (!exists)
	  continue;
	
	threadsFound++;
	
	cperfdmf_profile_t prof;
	
	if (CPERFDMF_OK == cperfdmf_getValues(&prof, selection, metric, n, c, t)) {
	  printf ("%d:%d:%d %9.2f %10.2f %10.2f %11.2f %8d %14d %11.2f\n",n,c,t,prof.exclusive / 1000, 
		  prof.inclusive / 1000, prof.exclusivePercent, prof.inclusivePercent,
		  prof.numCalls, prof.numSubroutines, prof.inclusivePerCall / 1000);
	} else {
	  printf ("%d:%d:%d ---\n", n, c, t);
	}
      }
    }
  }


  nodesFound = 0;

  for (n=0; n<numNodes; n++) {
    int exists;
    int ec;



    if (CPERFDMF_OK != (ec = cperfdmf_queryNodeExists(&exists, n))) {
      fprintf (stderr, "Uh oh: %s\n",cperfdmf_errToString(ec));
    }

    if (!exists)
      continue;

    nodesFound++;


    int numContexts;
    if (CPERFDMF_OK != (ce = cperfdmf_getNumContexts(&numContexts, n))) {
      fprintf (stderr, "error getting number of contexts: %s\n",cperfdmf_errToString(ce));
      if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
	fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
      }
      return -1;
    }

    
    int contextsFound = 0;
    
    for (c=0; contextsFound<numContexts; c++) {
      
      if (CPERFDMF_OK != (ec = cperfdmf_queryContextExists(&exists, n, c))) {
	fprintf (stderr, "%s(%d): Error %s\n", __FILE__, __LINE__, cperfdmf_errToString(ec));
      }

      if (!exists)
	continue;
      
      contextsFound++;

      int numThreads;
      if (CPERFDMF_OK != (ce = cperfdmf_getNumThreads(&numThreads, n, c))) {
	fprintf (stderr, "error getting number of threads: %s\n",cperfdmf_errToString(ce));
	if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
	  fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
	}
	return -1;
      }


      int threadsFound=0;
      for (t=0; threadsFound<numThreads; t++) {
	if (CPERFDMF_OK != (ec = cperfdmf_queryThreadExists(&exists, n, c, t))) {
	  fprintf (stderr, "%s(%d): Error %s\n", __FILE__, __LINE__, cperfdmf_errToString(ec));
	}
	
	if (!exists)
	  continue;
	
	threadsFound++;

	cperfdmf_userEvent_t *userEvents;
	int numUserEvents;
	int ce;
	//int ce = cperfdmf_getUserEvents(&userEvents, &numUserEvents, n, c, t);

	if (CPERFDMF_OK != (ce = cperfdmf_getUserEvents(&userEvents, &numUserEvents, n, c, t))) {
	  fprintf (stderr, "error getting user events: %s\n",cperfdmf_errToString(ce));
	  if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
	    fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
	  }
	  return;
	}

	int i;
	if (CPERFDMF_OK == ce) {

	  printf ("\nUSER EVENTS Profile :NODE %d, CONTEXT %d, THREAD %d\n",n,c,t);
	  printf ("---------------------------------------------------------------------------------------\n");
	  printf ("NumSamples   MaxValue   MinValue  MeanValue      SumSqr  Event Name\n");
	  printf ("---------------------------------------------------------------------------------------\n");

	  for (i = 0; i < numUserEvents; i++) {
	    printf ("%10d %10g %10g %10g %11g  %s\n", userEvents[i].numSamples, userEvents[i].maxValue,
		    userEvents[i].minValue, userEvents[i].meanValue, userEvents[i].sumSquared, userEvents[i].name);
	  }
	}
      }
    }
  }
  
  return 0;
}

int main(int argc, char** argv) {


  int i;
  int selection;
  int ce;
  char perfdmfcfg[2048];
  
  sprintf (perfdmfcfg,"%s/.ParaProf/perfdmf.cfg",getenv("HOME"));


  if (CPERFDMF_OK != (ce = cperfdmf_initialize(perfdmfcfg))) {
    fprintf (stderr, "initialization failed: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }

  
  /***********************************************************************
   * Here's how to load a trial from a TAU output rather than the database
   ***********************************************************************/

  /*
  if (CPERFDMF_OK != (ce = cperfdmf_loadTauOutput("/home/users/amorris/mpilieb/depth1"))) {
    fprintf (stderr, "TauOutput initialization failed: %s\n", cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }
  return displayTrialData();
  */

  /***********************************************************************/


  
  if (CPERFDMF_OK != (ce = cperfdmf_connectDB(NULL))) {
    fprintf (stderr, "db initialization failed: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }


  cperfdmf_application_t *apps;
  int numApps;

  if (CPERFDMF_OK != (ce = cperfdmf_getApplications(&apps, &numApps))) {
    printf ("Error loading applications: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }

  printf ("\nApplications:\n");
  for (i=0; i<numApps; i++) {
    printf ("%d) %s\n", i, apps[i].name);
  }

  printf ("Select an application: ");
  
  scanf ("%d",&selection);
  // sanity checks on input? no

  cperfdmf_experiment_t *exps;
  int numExps;

  if (CPERFDMF_OK != (ce = cperfdmf_getExperiments(&exps, &numExps, apps[selection].appId))) {
    printf ("Error loading experiments: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }

  cperfdmf_freeApplicationList(apps,numApps);

  if (numExps == 0) {
    printf ("no experiments for this application!\n");
    return 0;
  }
  printf ("\nExperiments:\n");
  for (i=0; i<numExps; i++) {
    printf ("%d) %s\n", i, exps[i].name);
  }

  printf ("Select an Experiment: ");
  scanf ("%d",&selection);
  // sanity checks on input? no



  cperfdmf_trial_t *trials;
  int numTrials;

  if (CPERFDMF_OK != (ce = cperfdmf_getTrials(&trials, &numTrials, exps[selection].expId))) {
    printf ("Error loading trials: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }


  if (numTrials == 0) {
    printf ("no trials for this experiment!\n");
    return 0;
  }

  printf("\nTrials:\n");
  for (i=0; i<numTrials; i++) {
    printf("%d) %s\n", i, trials[i].name);
  }

  printf("Select a Trial: ");
  scanf("%d",&selection);
  // sanity checks on input? no


  if (CPERFDMF_OK != (ce = cperfdmf_loadTrial(trials[selection].trialId))) {
    printf ("Error loading trial: %s\n",cperfdmf_errToString(ce));
    if (CPERFDMF_ERR_INTERNAL_ERROR == ce) {
      fprintf (stderr, "extended error: %s\n",cperfdmf_getExtendedError());
    }
    return -1;
  }

  ce = displayTrialData();

  cperfdmf_finalize();
  return ce;
}
