
/**
 * @file cperfdmf.c
 * The CPERFDMF (C api to PerfDMF) API
 *
 * @author Alan Morris
 */


#include <stdlib.h>
#include <stdio.h> 
#include <string.h>
#include "cperfdmf.h"
#include "jcperfdmf.h"
#include <stdarg.h>


static char extendedError[4096];
static char *theConfigFile;

void cperfdmf_setExtendedError(char *template, ...) {
  va_list ap;
  va_start (ap, template);
  vsnprintf (extendedError, 4096, template, ap);
  va_end (ap);
}

char* cperfdmf_getExtendedError() {
  return extendedError;
}


// this is here because strdup is not actually part of the C(99) library
static char* strduplicate (char *source) {
  char* retStr = (char*) malloc (sizeof(char) * (strlen(source) + 1));
  if (!retStr)
    return NULL;
  strcpy (retStr, source);
  return retStr;
}


// Possible return values 
// CPERFDMF_ERR_INTERNAL_ERROR
// CPERFDMF_ERR_OK
int getNextString(char** location, char* buffer, char* str, int* position) {
  int pos = *position;  // copy in the current position in 'str'
  int bufpos = 0;
  int done = 0;
  while (pos < strlen(str) && !done) {

    if (str[pos] == '\\') {  
      // skip over escape sequences (i.e. skip the \ and copy the 
      // next character, even if it is a semi-colon
      pos++;
      if (str[pos]) {
	buffer[bufpos++] = str[pos++];
      }
    } else {
      if (str[pos] == ';') 
	done = 1;
      else {
	buffer[bufpos++] = str[pos++];
      }
    }
  }

  if (!done) {
    // we reached the end of the string before finding a (non-escaped)
    // semi-colon, this is bad.
    cperfdmf_setExtendedError("%s(%d): Bad Parse, couldn't find a semi-colon\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  buffer[bufpos] = 0;  // null terminate

  // set the values to be returned
  *position = ++pos;  // skip over the semi-colon
  *location = strduplicate(buffer);  // copy out the string
  //  printf ("new ptr = 0x%x\n",*location);
  return CPERFDMF_OK;
}


// return values
// CPERFDMF_OK
// CPERFDMF_INTERNAL_ERROR
int parseStringList(char*** target, int* count, char* list) {
  if (list == NULL) {  // no items
    *target = NULL;
    *count = 0;
    return CPERFDMF_OK;
  }

  int pos = 0;
  int len = strlen(list);
  int numItems = 0;

  // first count the delimiters to see how many items there are
  while (pos < len) {
    if (list[pos] == '\\') {
      pos+=2;  // skip over the \ and the next char
    } else {
      if (list[pos] == ';') {
	numItems++;
      }
      pos++;
    }
  }

  if (numItems == 0) {  // nothing in the list
    *target = NULL;
    *count = 0;
    return CPERFDMF_OK;
  }

  char **itemList = (char **) malloc (sizeof(char *) * numItems);
  if (!itemList) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  int itemIdx = 0;
  pos = 0;

  char *buffer = (char *) malloc (sizeof(char) * strlen(list));
  if (!buffer) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }
  
  while (pos < len) {
    int ec;
    if (CPERFDMF_OK != (ec = getNextString(&itemList[itemIdx++], buffer, list, &pos))) 
      return ec;
  }

  *target = itemList;
  *count = numItems;
  return CPERFDMF_OK;
}


int parseIntegerList(int** target, int* count, char* list) {
  int pos, len, numItems;
  int* itemList;
  int itemIdx;
  int start, length;
  char *tmpstr;
  
  if (list == NULL) {
    *target = NULL;
    *count = 0;
    return CPERFDMF_OK;
  }

  pos = 0;
  len = strlen(list);
  numItems = 0;

  // first count the delimiters to see how many we've got
  while (pos < len) {
    if (list[pos] == ';') {
      numItems++;
    }
    pos++;
  }

  if (numItems == 0) {  // nothing in the list
    *target = NULL;
    *count = 0;
    return CPERFDMF_OK;
  }


  itemList = (int*) malloc (sizeof(int) * numItems);
  if (!itemList) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  
  itemIdx = 0;
  pos = 0;
  start = 0;
  while (pos < len) {

    while (pos < len && list[pos] != ';') {
      pos++;
    }

    // we are either at the end (pos == len) or at a delimiter

    length = pos - start;

    tmpstr = (char *) malloc (sizeof(char) * (length + 1));
    if (!tmpstr) {
      cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
      return CPERFDMF_ERR_INTERNAL_ERROR;
    }

    strncpy (tmpstr,&list[start],pos-start);
    tmpstr[length] = 0; // null terminate

    itemList[itemIdx] = atoi(tmpstr);
    free (tmpstr);
    itemIdx++;
    
    pos++; // skip over the delimiter
    start = pos;

  }
  
  *count = numItems;
  *target = itemList;

  return CPERFDMF_OK;
}



void writeNextString(char* buffer, int* position, char* str) {
  int pos = *position;  // copy in the current position

  if (str) {
    while (*str) {
      if (*str == '\\' || *str == ';') {
	buffer[pos++] = '\\';
      }
      
      if (*str)
	buffer[pos++] = *str++;
    }
  }

  buffer[pos++] = ';';

  // write back new position
  *position = pos;
}


void writeNextInteger(char* buffer, int* position, int integer) {
  *position += sprintf (buffer + *position, "%d;", integer);
}





// Possible return values 
// CPERFDMF_ERR_INTERNAL_ERROR
// CPERFDMF_ERR_OK
int getNextInteger(int* location, char* buffer, char* str, int* position) {
  char *tmpStr = NULL;
  // simply call getNextString and convert the output
  int retval = getNextString(&tmpStr, buffer, str, position);
  if (retval != CPERFDMF_OK)
    return retval;

  *location = atoi(tmpStr);  // copy out the integer
  free(tmpStr);
  return CPERFDMF_OK;
}

// Possible return values 
// CPERFDMF_ERR_INTERNAL_ERROR
// CPERFDMF_ERR_OK
int getNextDouble(double *location, char *buffer, char *str, int *position) {
  char *tmpStr = NULL;
  // simply call getNextString and convert the output
  int retval = getNextString(&tmpStr, buffer, str, position);
  if (retval != CPERFDMF_OK)
    return retval;

  *location = atof(tmpStr);  // copy out the double
  // printf ("double(%s) = %.20g\n",tmpStr,*location);
  free(tmpStr);
  return CPERFDMF_OK;
}


 

static int getApplicationDetail(cperfdmf_application_t* app, int appId, char** fieldNames, int numFields) {
  char *appDetail = NULL;
  int pos, ec, i;
  
  if (CPERFDMF_OK != (ec = jcperfdmf_getApplicationDetail(&appDetail, appId)))
    return ec;
  
  
  // parse the string into the 'cperfdmf_application_t' type
  char *buffer = (char *) malloc (sizeof(char) * strlen(appDetail));
  if (!buffer) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  // assign the fieldNames (note: every application has the same pointer since we only need to keep
  // one copy of the fieldNames
  app->numFields = numFields;
  app->fieldNames = fieldNames;

  app->fields = (char **) malloc (sizeof(char*) * numFields);
  if (!app->fields) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }
  pos = 0;
  
  if (CPERFDMF_OK != (ec=getNextString(&app->name, buffer, appDetail, &pos))) return ec;
  if (CPERFDMF_OK != (ec=getNextInteger(&app->appId, buffer, appDetail, &pos))) return ec;

  for (i=0; i < numFields; i++) {
    if (CPERFDMF_OK != (ec=getNextString(&app->fields[i], buffer, appDetail, &pos))) return ec;
  }
  
  free (buffer);
  return CPERFDMF_OK;
}

//////////////////////////////////////////////////////////////////////////
// API Functions
//////////////////////////////////////////////////////////////////////////



int strlenHelper(char *str) {
  if (!str)
    return 0;
  return strlen(str);
}

int cperfdmf_saveApplication(cperfdmf_application_t* app) {
  char* appDetail = NULL;
  int pos, ec, i;
  int stringSize = 2*strlenHelper(app->name) + 100; // name + id

  for (i = 0; i < app->numFields; i++) {
    stringSize += 2*strlenHelper(app->fields[i]);
  }
  appDetail = (char*) malloc (stringSize);
  if (!appDetail) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  pos = 0;

  writeNextString(appDetail, &pos, app->name);
  writeNextInteger(appDetail, &pos, app->appId);

  for (i = 0; i < app->numFields; i++) {
    writeNextString(appDetail, &pos, app->fields[i]);
  }
  
  appDetail[pos] = 0;  // null terminate
  int newId;

  ec = jcperfdmf_setApplicationDetail(appDetail, &newId);

  //printf ("new App id = %d\n", newId);
  app->appId = newId;

  free (appDetail);
  
  return ec;
}


int cperfdmf_saveExperiment(cperfdmf_experiment_t* exp) {
  char* expDetail = NULL;
  int pos, ec, i;

  int stringSize = 2*strlenHelper(exp->name) + 200; // name + appID + expID

  for (i = 0; i < exp->numFields; i++) {
    stringSize += 2*strlenHelper(exp->fields[i]);
  }

  expDetail = (char*) malloc (stringSize);

  if (!expDetail) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  pos = 0;

  writeNextInteger(expDetail, &pos, exp->appId);
  writeNextInteger(expDetail, &pos, exp->expId);
  writeNextString(expDetail, &pos, exp->name);

  for (i = 0; i < exp->numFields; i++) {
    writeNextString(expDetail, &pos, exp->fields[i]);
  }

  expDetail[pos] = 0;  // null terminate
  int newId;
  ec = jcperfdmf_setExperimentDetail(expDetail, &newId);

  exp->expId = newId;

  free (expDetail);
  return ec;
}



int cperfdmf_saveTrial(cperfdmf_trial_t* trial) {
  char* trialDetail = NULL;
  int pos, ec, i;

  int stringSize = 2*strlenHelper(trial->name) + 300; // name + appID + expID + trialID

  for (i = 0; i < trial->numFields; i++) {
    stringSize += 2*strlenHelper(trial->fields[i]);
  }

  trialDetail = (char*) malloc (stringSize);
  if (!trialDetail) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  pos = 0;

  writeNextInteger(trialDetail, &pos, trial->appId);
  writeNextInteger(trialDetail, &pos, trial->expId);
  writeNextInteger(trialDetail, &pos, trial->trialId);
  writeNextString(trialDetail, &pos, trial->name);

  for (i = 0; i < trial->numFields; i++) {
    writeNextString(trialDetail, &pos, trial->fields[i]);
  }


  trialDetail[pos] = 0;  // null terminate
  int newId;
  ec = jcperfdmf_setTrialDetail(trialDetail, &newId);

  trial->trialId = newId;

  free (trialDetail);
  return ec;

}



int cperfdmf_initialize(char *configFile) {
  // hold on to it
  theConfigFile = strduplicate(configFile);
  return jcperfdmf_init(theConfigFile);
}

void cperfdmf_finalize() {
  jcperfdmf_finalize();
}

int cperfdmf_connectDB(char *password) {
  return jcperfdmf_connectDB(theConfigFile, password);
}


int cperfdmf_loadTauOutput(char *path) {
  return jcperfdmf_initializeTauOutput(path);
}


int cperfdmf_deleteApplication(int appId) {
  return jcperfdmf_deleteApplication(appId);
}

int cperfdmf_deleteExperiment(int expId) {
  return jcperfdmf_deleteExperiment(expId);
}

int cperfdmf_deleteTrial(int trialId) {
  return jcperfdmf_deleteTrial(trialId);
}




void cperfdmf_freeApplicationList(cperfdmf_application_t* applications, int numApplications) {
  int i, j;
  for (i=0; i<numApplications; i++) {

    if (i==0) { // only free the field names once
      free (applications[i].fieldNames);
    }
    free (applications[i].name);

    for (j=0; j < applications[i].numFields; j++) {
      free (applications[i].fields[j]);
    }
  }
  free (applications);
}

void cperfdmf_freeExperimentList(cperfdmf_experiment_t* experiments, int numExperiments) {
  int i, j;
  for (i=0; i<numExperiments; i++) {

    if (i==0) {
      free (experiments[i].fieldNames);
    }
    free (experiments[i].name);

    for (j=0; j < experiments[i].numFields; j++) {
      free (experiments[i].fields[j]);
    }
    
  }
  free (experiments);
}

void cperfdmf_freeTrialList(cperfdmf_trial_t* trials, int numTrials) {
  int i, j;
  for (i=0; i<numTrials; i++) {

    if (i==0) {
      free (trials[i].fieldNames);
    }
    free (trials[i].name);

    for (j=0; j < trials[i].numFields; j++) {
      free (trials[i].fields[j]);
    }
    
  }
  free (trials);
}


int cperfdmf_getApplications(cperfdmf_application_t** applications, int* numApplications) {
  int ec;
  char* appIdsStr = NULL;

  if (CPERFDMF_OK != (ec = jcperfdmf_getApplications(&appIdsStr))) {
    return ec;
  }

  int* appIds = NULL;
  int numApps;

  ec = parseIntegerList(&appIds, &numApps, appIdsStr);
  free (appIdsStr);

  if (ec != CPERFDMF_OK) {
    return ec;
  }

  // now get the metadata
  char* metaDataStr = NULL;
  if (CPERFDMF_OK != (ec = jcperfdmf_getApplicationMetaData(&metaDataStr))) {
    return ec;
  }

  char **fieldNames;
  int numFields;

  if (CPERFDMF_OK != (ec = parseStringList(&fieldNames, &numFields, metaDataStr))) {
    return ec;
  }

  cperfdmf_application_t* apps = (cperfdmf_application_t *) malloc (sizeof(cperfdmf_application_t) * numApps);

  if (!apps) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  int i;
  for (i=0;i<numApps;i++) {
    if (CPERFDMF_OK != (ec = getApplicationDetail(&apps[i], appIds[i], fieldNames, numFields))) {
      return ec;
    }
  }

  *applications = apps;
  *numApplications = numApps;
  return CPERFDMF_OK;
}


int cperfdmf_getExperiments(cperfdmf_experiment_t** experiments, int* numExperiments, int appId) {
  
  int ec;
  char* expIdsStr = NULL;

  if (CPERFDMF_OK != (ec = jcperfdmf_getExperiments(&expIdsStr, appId))) {
    return ec;
  }

  int* expIds = NULL;
  int numExps;
  if (CPERFDMF_OK != (ec = parseIntegerList(&expIds, &numExps, expIdsStr))) {
    return ec;
  }


  // now get the metadata
  char* metaDataStr = NULL;
  if (CPERFDMF_OK != (ec = jcperfdmf_getExperimentMetaData(&metaDataStr))) {
    return ec;
  }

  char **fieldNames;
  int numFields;

  if (CPERFDMF_OK != (ec = parseStringList(&fieldNames, &numFields, metaDataStr))) {
    return ec;
  }

  cperfdmf_experiment_t* exps = (cperfdmf_experiment_t *) malloc (sizeof(cperfdmf_experiment_t) * numExps);
  if (!exps) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  int i, j;
  for (i=0;i<numExps;i++) {

    char *expDetail = NULL;
    if (CPERFDMF_OK != (ec = jcperfdmf_getExperimentDetail(&expDetail, expIds[i]))) {
      return ec;
    }
   
    // parse the string into the 'cperfdmf_experiment_t' type
    char *buffer = (char *) malloc (sizeof(char) * strlen(expDetail));
    if (!buffer) {
      cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
      return CPERFDMF_ERR_INTERNAL_ERROR;
    }


    // assign the fieldNames (note: every application has the same pointer since we only need to keep
    // one copy of the fieldNames
    exps[i].numFields = numFields;
    exps[i].fieldNames = fieldNames;
    
    exps[i].fields = (char **) malloc (sizeof(char*) * numFields);
    if (!exps[i].fields) {
      cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
      return CPERFDMF_ERR_INTERNAL_ERROR;
    }
    
    int pos = 0;

    if (CPERFDMF_OK != (ec=getNextInteger(&exps[i].appId, buffer, expDetail, &pos))) return ec;
    if (CPERFDMF_OK != (ec=getNextInteger(&exps[i].expId, buffer, expDetail, &pos))) return ec;
    if (CPERFDMF_OK != (ec=getNextString(&exps[i].name, buffer, expDetail, &pos))) return ec;

    for (j=0; j < numFields; j++) {
      if (CPERFDMF_OK != (ec=getNextString(&exps[i].fields[j], buffer, expDetail, &pos))) return ec;
    }

    free (buffer);
  }

  *experiments = exps;
  *numExperiments = numExps;
  return CPERFDMF_OK;
}


int cperfdmf_getTrials(cperfdmf_trial_t** trials, int* numTrials, int expId) {
  int ec;

  char* trialIdsStr = NULL;

  if (CPERFDMF_OK != (ec = jcperfdmf_getTrials(&trialIdsStr, expId))) {
    return ec;
  }

  int numTrls;
  int* trialIds = NULL;

  if (CPERFDMF_OK != (ec = parseIntegerList(&trialIds, &numTrls, trialIdsStr))) {
    return ec;
  }



  // now get the metadata
  char* metaDataStr = NULL;
  if (CPERFDMF_OK != (ec = jcperfdmf_getTrialMetaData(&metaDataStr))) {
    return ec;
  }

  char **fieldNames;
  int numFields;

  if (CPERFDMF_OK != (ec = parseStringList(&fieldNames, &numFields, metaDataStr))) {
    return ec;
  }



  cperfdmf_trial_t *trls = (cperfdmf_trial_t *) malloc (sizeof(cperfdmf_trial_t) * numTrls);
  if (!trls) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  int i, j;

  for (i=0;i<numTrls;i++) {

    char* trialDetail = NULL;
    if (CPERFDMF_OK != 
	(ec = jcperfdmf_getTrialDetail(&trialDetail, trialIds[i]))) {
      return ec;
    }

    //printf ("got back trial detail '%s'\n",trialDetail);
  
    // parse the string into the 'trial' type
    char *buffer = (char *) malloc (sizeof(char) * strlen(trialDetail));
    if (!buffer) {
      cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
      return CPERFDMF_ERR_INTERNAL_ERROR;
    }


    // assign the fieldNames (note: every application has the same pointer since we only need to keep
    // one copy of the fieldNames
    trls[i].numFields = numFields;
    trls[i].fieldNames = fieldNames;


    trls[i].fields = (char **) malloc (sizeof(char*) * numFields);
    if (!trls[i].fields) {
      cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
      return CPERFDMF_ERR_INTERNAL_ERROR;
    }

    int pos = 0;

    int rt;
    if (CPERFDMF_OK != (rt=getNextInteger(&trls[i].appId, buffer, trialDetail, &pos))) return rt;
    if (CPERFDMF_OK != (rt=getNextInteger(&trls[i].expId, buffer, trialDetail, &pos))) return rt;
    if (CPERFDMF_OK != (rt=getNextInteger(&trls[i].trialId, buffer, trialDetail, &pos))) return rt;
    if (CPERFDMF_OK != (rt=getNextString(&trls[i].name, buffer, trialDetail, &pos))) return rt;


    for (j=0; j < numFields; j++) {
      if (CPERFDMF_OK != (ec=getNextString(&trls[i].fields[j], buffer, trialDetail, &pos))) return ec;
    }
    
    free (buffer);
  }

  *trials = trls;
  *numTrials = numTrls;

  return CPERFDMF_OK;
}


int cperfdmf_loadTrial(int trialId) {
  return jcperfdmf_selectTrial(trialId);
}


int cperfdmf_getMetrics(char*** metrics, int* numMetrics) {

  int ec;
  char* metricStr = NULL;
  if (CPERFDMF_OK != (ec = jcperfdmf_getMetrics(&metricStr))) {
    return ec;
  }

  // careful, parseStringList modifies metrics and numMetrics
  return parseStringList(metrics, numMetrics, metricStr); 
}


int cperfdmf_getFunctions(char*** functions, int* numFunctions) {
  
  int ec;
  char* functionStr = NULL;

  if (CPERFDMF_OK != (ec = jcperfdmf_getFunctions(&functionStr))) {
    return ec;
  }

  // careful, parseStringList modifies functions and numFunctions
  return parseStringList(functions, numFunctions, functionStr);
}


int cperfdmf_getGroups(char*** groups, int* numGroups) {
  int ec;
  char* groupStr = NULL;

  if (CPERFDMF_OK != (ec = jcperfdmf_getGroups(&groupStr))) {
    return ec;
  }

  //printf ("got groups: %s\n", groupStr);
  ec = parseStringList(groups, numGroups, groupStr);
  free (groupStr);
  return ec;
}



int cperfdmf_getValues(cperfdmf_profile_t* prof, int functionId, int metric, int node, int context, int thread) {
  
  int ec;
  char* profileStr = NULL;
  char* buffer = NULL;
  char* groupStr = NULL;
  int i, pos;

  if (CPERFDMF_OK != (ec = jcperfdmf_getProfileValues(&profileStr, functionId, metric, node, context, thread))) {
    return ec;
  }

  if (profileStr == NULL) {
    return CPERFDMF_NON_EXISTENT;
  }

  //  printf ("got back profile str: %s\n", profileStr);

  // parse the string into the 'profile' type
  buffer = (char*) malloc (sizeof(char) * strlen(profileStr));
  if (!buffer) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }
  
  pos = 0;

  getNextDouble(&prof->exclusive, buffer, profileStr, &pos);
  getNextDouble(&prof->inclusive, buffer, profileStr, &pos);
  getNextDouble(&prof->exclusivePercent, buffer, profileStr, &pos);
  getNextDouble(&prof->inclusivePercent, buffer, profileStr, &pos);
  getNextDouble(&prof->inclusivePerCall, buffer, profileStr, &pos);
  getNextInteger(&prof->numCalls, buffer, profileStr, &pos);
  getNextInteger(&prof->numSubroutines, buffer, profileStr, &pos);
  getNextString(&groupStr, buffer, profileStr, &pos);
  
  // replace '@' with ';' in the group list, now we can use our
  // regular parse function to get the list of groups;
  for (i=0; i<strlen(groupStr); i++) {
    if (groupStr[i] == '@') {
      groupStr[i] = ';';
    }
  }

  if (CPERFDMF_OK != (ec = parseIntegerList(&prof->groups, &prof->numGroups, groupStr))) {
    free (groupStr);
    free (buffer);
    return ec;
  }

  free (groupStr);
  free (buffer);

  return CPERFDMF_OK;
}


int cperfdmf_queryNodeExists(int* exists, int node) {
  return jcperfdmf_queryNodeExists(exists, node);
}
int cperfdmf_queryContextExists(int* exists, int node, int context) {
  return jcperfdmf_queryContextExists(exists, node, context);
}
int cperfdmf_queryThreadExists(int* exists, int node, int context, int thread) {
  return jcperfdmf_queryThreadExists(exists, node, context, thread);
}


int cperfdmf_getUserEvents(cperfdmf_userEvent_t **userEvents, int *numUserEvents, 
			     int node, int context, int thread) {

  int ec;
  char* userEventStr = NULL;

  if (CPERFDMF_OK != 
      (ec = jcperfdmf_getUserEvents(&userEventStr, node, context, thread))) {
    return ec;
  }

  //printf ("userEventStr = %s\n",userEventStr);

  if (userEventStr == NULL) {
    return CPERFDMF_NON_EXISTENT;
  }

  char *buffer = (char*) malloc (sizeof(char) * strlen(userEventStr));
  if (!buffer) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }

  int pos = 0;

  // the first element in the string is the number of events to follow
  int numEvents;
  if (CPERFDMF_OK != (ec = getNextInteger(&numEvents, buffer, userEventStr, &pos))) {
    return ec;
  }
  
  cperfdmf_userEvent_t *events = (cperfdmf_userEvent_t*) malloc (sizeof(cperfdmf_userEvent_t) * numEvents);

  if (!events) {
    cperfdmf_setExtendedError("%s(%d): malloc failed\n", __FILE__, __LINE__);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }
  
  // grab each event
  int i;
  for (i = 0; i < numEvents; i++) {
    getNextString(&events[i].name, buffer, userEventStr, &pos);
    getNextInteger(&events[i].numSamples, buffer, userEventStr, &pos);
    getNextDouble(&events[i].minValue, buffer, userEventStr, &pos);
    getNextDouble(&events[i].maxValue, buffer, userEventStr, &pos);
    getNextDouble(&events[i].meanValue, buffer, userEventStr, &pos);
    getNextDouble(&events[i].sumSquared, buffer, userEventStr, &pos);
  }

  free (userEventStr);

  // set the values to be returned
  *userEvents = events;
  *numUserEvents = numEvents;
  return CPERFDMF_OK;
}

int cperfdmf_getMeanValues(cperfdmf_profile_t* prof, int functionId, int metric) {
  return cperfdmf_getValues(prof, functionId, metric, -1, -1, -1);
}

int cperfdmf_getTotalValues(cperfdmf_profile_t* prof, int functionId, int metric) {
  return cperfdmf_getValues(prof, functionId, metric, -2, -2, -2);
}




int cperfdmf_getNumNodes(int* numNodes) {
  return jcperfdmf_getNumNodes(numNodes);
}

int cperfdmf_getNumContexts(int* numContexts, int node) {
  return jcperfdmf_getNumContexts(numContexts, node);
}

int cperfdmf_getNumThreads(int* numThreads, int node, int context) {
  return jcperfdmf_getNumThreads(numThreads, node, context);
}




char *cperfdmf_errToString(int code) {
  switch (code) {
  case CPERFDMF_OK: return "CPERFDMF_OK";
  case CPERFDMF_ERR: return "CPERFDMF_ERR";
  case CPERFDMF_NON_EXISTENT: return "CPERFDMF_NON_EXISTENT";
  case CPERFDMF_ERR_NO_PROFILES: return "CPERFDMF_ERR_NO_PROFILES";
  case CPERFDMF_ERR_INTERNAL_ERROR: return "CPERFDMF_ERR_INTERNAL_ERROR";
  case CPERFDMF_ERR_DB_NOT_INITIALIZED: return "CPERFDMF_ERR_DB_NOT_INITIALIZED";
  case CPERFDMF_ERR_SESSION_NOT_INITIALIZED: return "CPERFDMF_ERR_SESSION_NOT_INITIALIZED";
  case CPERFDMF_ERR_JAVA_EXCEPTION: return "CPERFDMF_ERR_JAVA_EXCEPTION";
  }
  return "Unknown";
}

int cperfdmf_uploadTauOutput(int* trialId, char* path, int experimentId) {
  return jcperfdmf_uploadTauOutput(trialId, path, experimentId);
}
