
/**
 * @file jcperfdmf.c
 * The CPERFDMF (C api to PerfDMF) API
 *
 * @author Alan Morris
 */


#include <stdlib.h>
#include <stdlib.h>
#include <string.h>
#include "cperfdmf.h"
#include "jni.h"

#define JNI_VERSION_1_1 0x00010001
#define JNI_VERSION_1_2 0x00010002

// do some ifdef windows?
#define PATH_SEPARATOR ":"
//#define USER_CLASSPATH ""


#ifndef JAR_LOCATION
#error JAR_LOCATION must be defined
#endif

JavaVM *jvm = NULL;
JNIEnv *env = NULL;

jobject cperfdmfObject;
jclass cperfdmfClass;
jclass stringHolderClass;
jclass integerHolderClass;

jmethodID integerHolderClass_ctor;
jmethodID stringHolderClass_ctor;
jmethodID integerHolderClass_get;
jmethodID stringHolderClass_get;


extern void cperfdmf_setExtendedError(char *template, ...);




// this is here because strdup is not actually part of the C(99) library
static char* strduplicate (const char *source) {
  char* retStr = (char*) malloc (sizeof(char) * (strlen(source) + 1));
  if (!retStr)
    return NULL;
  strcpy (retStr, source);
  return retStr;
}

static jstring javaString(char *str) {
  jstring jstr;

  jstr = (*env)->NewStringUTF(env, str);
  if (jstr == 0) {
    cperfdmf_setExtendedError("%s(%d): NewStringUTF failed\n", __FILE__, __LINE__);
  }
  return jstr;
}

static char* j2cString(jstring jstr) {
  char* string = NULL;
  if (jstr != (jstring) NULL) {
    const char* utf = (*env)->GetStringUTFChars(env, jstr, NULL);
    string = strduplicate(utf);
    (*env)->ReleaseStringUTFChars(env, jstr, utf);
  }

  return string;
}


static int getJavaException() {
  jthrowable except;
  jstring msg;
  char* message;
  
  //    (*env)->ExceptionDescribe(env);
  if (0 != (except = (*env)->ExceptionOccurred(env))) {
    (*env)->ExceptionClear(env);
    
    msg = (jstring) (*env)->CallObjectMethod(env, except, (*env)->GetMethodID(env, (*env)->FindClass (env, "java/lang/Throwable"), "toString", "()Ljava/lang/String;"));
    
    message = j2cString(msg);
    cperfdmf_setExtendedError("Java Exception: %s\n", message);
    free (message);
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }
  return CPERFDMF_ERR;
}

int jcperfdmf_init(char *configFile) {

  JavaVMInitArgs vm_args;
  jint res;
  char *classpath;
  JavaVMOption options[2];
  FILE *file;
  char linebuf[2048];
  char *retVal;
  char *jdbcJarFile;
  int i;

  file = fopen(configFile, "r");
  

  retVal = fgets (linebuf, 2048, file);
  while (retVal != NULL) {
    if (strncmp(linebuf,"jdbc_db_jarfile:",16) == 0) {
      jdbcJarFile=&(linebuf[15]);

      // null terminate it
      for (i=15;i<2048;i++) {
	if (linebuf[i] == '\n' || linebuf[i] == '\r') {
	  linebuf[i] = 0;
	}
      }
      break;
    }
    
    retVal = fgets (linebuf, 2048, file);
  }

/*   printf ("got jar:'%s'\n",jdbcJarFile); */

  if (getenv("CLASSPATH")) {
    classpath = (char*) malloc (sizeof(char) * (25 + strlen( (char*) getenv("CLASSPATH")) + strlen(jdbcJarFile)+strlen(JAR_LOCATION)));
    sprintf (classpath, "-Djava.class.path=%s%s%s%s%s", getenv("CLASSPATH"), PATH_SEPARATOR, jdbcJarFile, PATH_SEPARATOR, JAR_LOCATION);
  } else {
    classpath = (char*) malloc (sizeof(char) * (25 + strlen(jdbcJarFile)+strlen(JAR_LOCATION)));
    sprintf (classpath, "-Djava.class.path=%s%s%s", jdbcJarFile, PATH_SEPARATOR, JAR_LOCATION);
  }


  options[0].optionString = "-Djava.compiler=NONE";
  options[1].optionString = classpath;
  options[2].optionString = "-Xmx500m";


  JNI_GetDefaultJavaVMInitArgs(&vm_args);
  
  /* IMPORTANT: specify vm_args version # if you use JDK1.1.2 and beyond */
  vm_args.version = JNI_VERSION_1_2;
  vm_args.options = options;
  vm_args.nOptions = 3;
  vm_args.ignoreUnrecognized = 1;
  
  /* Create the Java VM */
  res = JNI_CreateJavaVM(&jvm, (void**) &env, &vm_args);
  if (res < 0) {
    return CPERFDMF_ERR;
  }

  cperfdmfClass = (*env)->FindClass(env, "Cperfdmf");
  if (cperfdmfClass == 0) {
    return getJavaException();
  }


  jmethodID ctor = (*env)->GetMethodID(env, cperfdmfClass, "<init>", "()V");

  cperfdmfObject = (*env)->NewObject(env, cperfdmfClass, ctor);
  if (cperfdmfObject == 0) {
    return CPERFDMF_ERR;
  }

  integerHolderClass = (*env)->FindClass(env, "IntegerHolder");
  if (integerHolderClass == 0) {
    return CPERFDMF_ERR;
  }

  stringHolderClass = (*env)->FindClass(env, "StringHolder");
  if (stringHolderClass == 0) {
    return CPERFDMF_ERR;
  }

  integerHolderClass_ctor = (*env)->GetMethodID(env, integerHolderClass, "<init>", "()V");
  stringHolderClass_ctor = (*env)->GetMethodID(env, stringHolderClass, "<init>", "()V");

  integerHolderClass_get = (*env)->GetMethodID(env, integerHolderClass, "get", "()I");
  stringHolderClass_get = (*env)->GetMethodID(env, stringHolderClass, "get", "()Ljava/lang/String;");

  // check those

  return CPERFDMF_OK;
};

int jcperfdmf_finalize() {
  (*jvm)->DestroyJavaVM(jvm);
}

int exceptionCheck(int code) {
  /*  if (code == CPERFDMF_ERR_JAVA_EXCEPTION) {
    cperfdmf_setExtendedError("Unhandled Java Exception: %s", cperfdmf_dmf_getExceptionMessage(h));
    return CPERFDMF_ERR_INTERNAL_ERROR;
  }
  return code;
  */
  jmethodID mid;
  jstring jstr;

  if (code == CPERFDMF_ERR_JAVA_EXCEPTION) {
  
    mid = (*env)->GetMethodID(env, cperfdmfClass, "getExceptionMessage", "()Ljava/lang/String;");
    if (mid == 0) {
      cperfdmf_setExtendedError("Unable to location getExceptionMessage method");
      return CPERFDMF_ERR_INTERNAL_ERROR;
    }
    

    jstr = (jstring) (*env)->CallObjectMethod(env, cperfdmfObject, mid);

    
    if (jstr != (jstring) NULL) {
      const char* utf = (*env)->GetStringUTFChars(env, jstr, NULL);
      cperfdmf_setExtendedError("Unhandled Java Exception: %s", utf);
      (*env)->ReleaseStringUTFChars(env, jstr, utf);
    }

    return CPERFDMF_ERR_INTERNAL_ERROR;
  }
  return code;

}






int jcperfdmf_connectDB(char* arg0, char* arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "connectDB", "(Ljava/lang/String;Ljava/lang/String;)I");
	jstring jstr0 = javaString(arg0);
	jstring jstr1 = javaString(arg1);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, jstr0, jstr1)))) {
		return ec;
	}
	return CPERFDMF_OK;
}

int jcperfdmf_initializeTauOutput(char* arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "initializeTauOutput", "(Ljava/lang/String;)I");
	jstring jstr0 = javaString(arg0);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, jstr0)))) {
		return ec;
	}
	return CPERFDMF_OK;
}

int jcperfdmf_uploadTauOutput(int* arg0, char* arg1, int arg2) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "uploadTauOutput", "(LIntegerHolder;Ljava/lang/String;I)I");
	jobject integerHolder0 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	jstring jstr1 = javaString(arg1);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, integerHolder0, jstr1, arg2)))) {
		return ec;
	}
	*arg0 = (*env)->CallIntMethod(env, integerHolder0, integerHolderClass_get);
	return CPERFDMF_OK;
}

int jcperfdmf_getApplications(char** arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getApplications", "(LStringHolder;)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getApplicationMetaData(char** arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getApplicationMetaData", "(LStringHolder;)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getApplicationDetail(char** arg0, int arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getApplicationDetail", "(LStringHolder;I)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0, arg1)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_setApplicationDetail(char* arg0, int* arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "setApplicationDetail", "(Ljava/lang/String;LIntegerHolder;)I");
	jstring jstr0 = javaString(arg0);
	jobject integerHolder1 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, jstr0, integerHolder1)))) {
		return ec;
	}
	*arg1 = (*env)->CallIntMethod(env, integerHolder1, integerHolderClass_get);
	return CPERFDMF_OK;
}

int jcperfdmf_deleteApplication(int arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "deleteApplication", "(I)I");
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, arg0)))) {
		return ec;
	}
	return CPERFDMF_OK;
}

int jcperfdmf_getExperiments(char** arg0, int arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getExperiments", "(LStringHolder;I)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0, arg1)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getExperimentMetaData(char** arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getExperimentMetaData", "(LStringHolder;)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getExperimentDetail(char** arg0, int arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getExperimentDetail", "(LStringHolder;I)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0, arg1)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_setExperimentDetail(char* arg0, int* arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "setExperimentDetail", "(Ljava/lang/String;LIntegerHolder;)I");
	jstring jstr0 = javaString(arg0);
	jobject integerHolder1 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, jstr0, integerHolder1)))) {
		return ec;
	}
	*arg1 = (*env)->CallIntMethod(env, integerHolder1, integerHolderClass_get);
	return CPERFDMF_OK;
}

int jcperfdmf_deleteExperiment(int arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "deleteExperiment", "(I)I");
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, arg0)))) {
		return ec;
	}
	return CPERFDMF_OK;
}

int jcperfdmf_getTrials(char** arg0, int arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getTrials", "(LStringHolder;I)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0, arg1)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getTrialMetaData(char** arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getTrialMetaData", "(LStringHolder;)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getTrialDetail(char** arg0, int arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getTrialDetail", "(LStringHolder;I)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0, arg1)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_setTrialDetail(char* arg0, int* arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "setTrialDetail", "(Ljava/lang/String;LIntegerHolder;)I");
	jstring jstr0 = javaString(arg0);
	jobject integerHolder1 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, jstr0, integerHolder1)))) {
		return ec;
	}
	*arg1 = (*env)->CallIntMethod(env, integerHolder1, integerHolderClass_get);
	return CPERFDMF_OK;
}

int jcperfdmf_deleteTrial(int arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "deleteTrial", "(I)I");
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, arg0)))) {
		return ec;
	}
	return CPERFDMF_OK;
}

int jcperfdmf_selectTrial(int arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "selectTrial", "(I)I");
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, arg0)))) {
		return ec;
	}
	return CPERFDMF_OK;
}

int jcperfdmf_getMetrics(char** arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getMetrics", "(LStringHolder;)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getFunctions(char** arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getFunctions", "(LStringHolder;)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getNumNodes(int* arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getNumNodes", "(LIntegerHolder;)I");
	jobject integerHolder0 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, integerHolder0)))) {
		return ec;
	}
	*arg0 = (*env)->CallIntMethod(env, integerHolder0, integerHolderClass_get);
	return CPERFDMF_OK;
}

int jcperfdmf_getNumContexts(int* arg0, int arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getNumContext", "(LIntegerHolder;I)I");
	jobject integerHolder0 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, integerHolder0, arg1)))) {
		return ec;
	}
	*arg0 = (*env)->CallIntMethod(env, integerHolder0, integerHolderClass_get);
	return CPERFDMF_OK;
}

int jcperfdmf_getNumThreads(int* arg0, int arg1, int arg2) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getNumThreads", "(LIntegerHolder;II)I");
	jobject integerHolder0 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, integerHolder0, arg1, arg2)))) {
		return ec;
	}
	*arg0 = (*env)->CallIntMethod(env, integerHolder0, integerHolderClass_get);
	return CPERFDMF_OK;
}

int jcperfdmf_getProfileValues(char** arg0, int arg1, int arg2, int arg3, int arg4, int arg5) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getProfileValues", "(LStringHolder;IIIII)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0, arg1, arg2, arg3, arg4, arg5)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getUserEvents(char** arg0, int arg1, int arg2, int arg3) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getUserEvents", "(LStringHolder;III)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0, arg1, arg2, arg3)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_getGroups(char** arg0) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "getGroups", "(LStringHolder;)I");
	jobject stringHolder0 = (*env)->NewObject(env, stringHolderClass, stringHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, stringHolder0)))) {
		return ec;
	}
	jstring jstr0 = (jstring) (*env)->CallObjectMethod(env, stringHolder0, stringHolderClass_get);
	*arg0 = j2cString(jstr0);
	return CPERFDMF_OK;
}

int jcperfdmf_queryNodeExists(int* arg0, int arg1) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "queryNodeExists", "(LIntegerHolder;I)I");
	jobject integerHolder0 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, integerHolder0, arg1)))) {
		return ec;
	}
	*arg0 = (*env)->CallIntMethod(env, integerHolder0, integerHolderClass_get);
	return CPERFDMF_OK;
}

int jcperfdmf_queryContextExists(int* arg0, int arg1, int arg2) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "queryContextExists", "(LIntegerHolder;II)I");
	jobject integerHolder0 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, integerHolder0, arg1, arg2)))) {
		return ec;
	}
	*arg0 = (*env)->CallIntMethod(env, integerHolder0, integerHolderClass_get);
	return CPERFDMF_OK;
}

int jcperfdmf_queryThreadExists(int* arg0, int arg1, int arg2, int arg3) {
	jmethodID mid = (*env)->GetMethodID(env, cperfdmfClass, "queryThreadExists", "(LIntegerHolder;III)I");
	jobject integerHolder0 = (*env)->NewObject(env, integerHolderClass, integerHolderClass_ctor);
	int ec;
	if (CPERFDMF_OK != (ec = exceptionCheck((*env)->CallIntMethod(env, cperfdmfObject, mid, integerHolder0, arg1, arg2, arg3)))) {
		return ec;
	}
	*arg0 = (*env)->CallIntMethod(env, integerHolder0, integerHolderClass_get);
	return CPERFDMF_OK;
}

