/**
 * @file cperfdmf.h
 * The CPERFDMF (C api to PerfdMf) API
 *
 * @author Alan Morris
 * 


 Using the C api to PerfDMF:

 @li Call cperfdmf_initialize()
 
 @li To get performance data from the database, use:
 cperfdmf_connectDB(), cperfdmf_getApplications(), cperfdmf_getExperiments(), cperfdmf_getTrials(), and cperfdmf_loadTrial()

 @li To get performance data directly from raw TAU profile.* files, use cperfdmf_loadTauOutput()

 @li After preparing your session with either cperfdmf_loadTrial(), or cperfdmf_TauOutput(), the rest of the functions will provide access to the performance data.

 @li All functions return the int enumeration.  You can convert this to a string for display, use cperfdmf_errToString().  In the event that CPERFDMF_ERR_INTERNAL_ERROR is returned, use cperfdmf_getExtendedError() for more details. 

 @see testApp.c

*/



#ifndef __CPERFDMF_H__
#define __CPERFDMF_H__



/**
 * Holds all metadata regarding a single trial from the database
 */
typedef struct cperfdmf_trial_t {
  /* /// unique (database-wide) application identifier */
  int appId;  
  /* /// unique (database-wide) experiment identifier */
  int expId;
  /* /// unique (database-wide) trial identifier */
  int trialId;
  char *name;

  char **fieldNames;
  char **fields;
  int numFields;
} cperfdmf_trial_t;


/**
 * Holds all metadeta concerning a single experiment from the database
 */
typedef struct cperfdmf_experiment_t {
  int appId;
  int expId;
  char *name;

  char **fieldNames;
  char **fields;
  int numFields;
} cperfdmf_experiment_t;


/**
 * Holds all metadeta concerning a single application from the database
 */
typedef struct cperfdmf_application_t {
  char *name;
  int appId;

  char **fieldNames;
  char **fields;
  int numFields;
} cperfdmf_application_t;


/**
 * Holds the actual values for a timer (function)
 */
typedef struct cperfdmf_profile_t {
  double exclusive;
  double exclusivePercent;
  double inclusive;
  double inclusivePerCall;
  double inclusivePercent;
  int numCalls;
  int numSubroutines;
  int *groups;
  int numGroups;
  /*
    int context;
    int node;
    int numCalls;
    int numSubroutines;
    int thread;
  */
} cperfdmf_profile_t;


/**
 * Holds the data for TAU user events
 */
typedef struct cperfdmf_userEvent_t {
  char* name;
  int numSamples;
  double maxValue;
  double minValue;
  double meanValue;
  double sumSquared;
} cperfdmf_userEvent_t;


/**
 * @enum int
 * Return values for cperfdmf_* functions.  In the event that CPERFDMF_ERR_INTERNAL_ERROR 
 * is returned, use cperfdmf_getExtendedError() to get more information 
 * on what happened.
 */
typedef enum {
  CPERFDMF_OK           =  0,
  CPERFDMF_ERR          = -1,
  CPERFDMF_NON_EXISTENT = -2,
  CPERFDMF_ERR_NO_PROFILES  = -3,
  CPERFDMF_ERR_INTERNAL_ERROR = -6,
  CPERFDMF_ERR_DB_NOT_INITIALIZED = -7,
  CPERFDMF_ERR_SESSION_NOT_INITIALIZED = -8,
  CPERFDMF_ERR_JAVA_EXCEPTION = -9
} cperfdmf_error_t;


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

  /**
   * @fn int cperfdmf_initialize(char *configFile)
   * initialize the cperfdmf system
   *
   * @param configFile [in] location of perfdmf.cfg (e.g. /home/users/amorris/.ParaProf/perfdmf.cfg)
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR failure
   */
  int cperfdmf_initialize(char *configFile);
  
  /**
   * @fn void cperfdmf_finalize()
   * finalize the cperfdmf system
   */
  void cperfdmf_finalize();
  
  
  
  
  /**
   * @defgroup DBstuff Database Functions
   */
  /*@{*/

  /**
   * @fn int cperfdmf_connectDB(char *password)
   * Connect to the database
   * 
   * @param password [in] password for user given in configFile, if you want to use the 
   *                      db_password from the configFile, set the password to NULL. 
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_INTERNAL_ERROR something went wrong (bad password?), use cperfdmf_getExtendedError() for details
   */
  int cperfdmf_connectDB(char *password);


  /**
   * @fn int cperfdmf_getApplications(cperfdmf_application_t** applications, int* numApplications)
   * Retrieve the applications from the database
   *
   * @param applications [out] pointer to an array of type cperfdmf_application_t.  
   *   The array will be allocated and generated by the API.  The user may free it at any time.
   * @param numApplications [out] pointer to an integer.  
   *   The value will be written with the number of applications returned.
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use cperfdmf_getExtendedError() for details
   */
  int cperfdmf_getApplications(cperfdmf_application_t** applications, int* numApplications);


  /**
   * @fn int cperfdmf_getExperiments(cperfdmf_experiment_t** experiments, int* numExperiments, int applicationId)
   * Retrieve the experiments for a given application
   *
   * @param experiments [out] pointer to an array of type cperfdmf_experiment_t.  
   *   The array will be allocated and generated by the API.  The user may free it at any time.
   * @param numExperiments [out] pointer to an integer.  The value will be written with the 
   *   number of applications returned.
   * @param applicationId [in] integer. The Identifier of application from which to load the exteriments.
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getExperiments(cperfdmf_experiment_t** experiments, int* numExperiments, int applicationId);


  /**
   * @fn int cperfdmf_getTrials(cperfdmf_trial_t** trials, int *numTrials, int experimentId)
   * Retrieve the trials for a given experiment
   *
   * @param trials [out] pointer to an array of type cperfdmf_trial_t.
   *   The array will be allocated and generated by the API.  The user may free it at any time.
   * @param numTrials [out] pointer to an integer.  The value will be written wit the
   *   number of trials returned.
   * @param experimentId [in] integer. The identifier of the experiment from which to load the trials.
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getTrials(cperfdmf_trial_t** trials, int *numTrials, int experimentId);


  /**
   * @fn int cperfdmf_loadTrial(int trialId)
   * Set the trial for subsequent calls
   *
   * @param trialId [in] integer. The identifier of the trial to be selected.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_loadTrial(int trialId);


  /**
   * @fn int cperfdmf_saveApplication(cperfdmf_application_t* application)
   * Save application metadata to the database
   *
   * @param application [in/out] pointer to a cperfdmf_application_t.  To create a new
   *        application, set the id (appId) to -1, after the call appId will be set to
   *        the id stored in the database.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_saveApplication(cperfdmf_application_t* application);



  /**
   * @fn int cperfdmf_saveExperiment(cperfdmf_experiment_t* experiment)
   * Save experiment metadata to the database
   *
   * @param experiment [in/out] pointer to a cperfdmf_experiment_t.  To create a new
   *        experiment, set the id (expId) to -1, after the call expId will be set to
   *        the id stored in the database.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_saveExperiment(cperfdmf_experiment_t* experiment);



  /**
   * @fn int cperfdmf_saveTrial(cperfdmf_trial_t* trial)
   * Save trial metadata to the database
   *
   * @param trial [in/out] pointer to a cperfdmf_trial_t. 
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_saveTrial(cperfdmf_trial_t* trial);


  /**
   * @fn int cperfdmf_uploadTauOutput(int* trialId, char* path, int experimentId)
   * Upload a TAU output session (profile.*.*.*) to the database
   *
   * @param trialId [out] pointer to an integer.  Will be filled in with the ID of the
   *                      trial created.
   * @param path [in] directory containing profile files
   * @param experimentId [in] Which experiment to associate this trial with.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_uploadTauOutput(int* trialId, char* path, int experimentId);

  
  /**
   * @fn int cperfdmf_deleteApplication(int applicationId)
   * Delete application from the database
   *
   * @param applicationId [in] integer. The identifier of the application to be deleted.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_deleteApplication(int applicationId);

  /**
   * @fn int cperfdmf_deleteExperiment(int experimentId)
   * Delete experiment from the database
   *
   * @param experimentId [in] integer. The identifier of the experiment to be deleted.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_deleteExperiment(int experimentId);

  /**
   * @fn int cperfdmf_deleteExperiment(int experimentId)
   * Delete experiment from the database
   *
   * @param experimentId [in] integer. The identifier of the experiment to be deleted.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_DB_NOT_INITIALIZED the database has not been initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_deleteTrial(int trialId);


  /**
   * @fn void cperfdmf_freeApplicationList(cperfdmf_application_t* applications, int numApplications); 
   * Helper function to free an array of applications
   *
   * @param applications [in] array of cperfdmf_application_t to be free'd.
   * @param numApplications [in] integer, number of applications to be free'd.
   * 
   */
  void cperfdmf_freeApplicationList(cperfdmf_application_t* applications, int numApplications); 


  /**
   * @fn void cperfdmf_freeExperimentList(cperfdmf_experiment_t* experiments, int numExperiments); 
   * Helper function to free an array of experiments
   *
   * @param experiments [in] array of cperfdmf_experiment_t to be free'd.
   * @param numExperiments [in] integer, number of experiments to be free'd.
   * 
   */
  void cperfdmf_freeExperimentList(cperfdmf_experiment_t* experiments, int numExperiments); 


  /**
   * @fn void cperfdmf_freeTrialList(cperfdmf_trial_t* trials, int numTrials); 
   * Helper function to free an array of trials
   *
   * @param trials [in] array of cperfdmf_trial_t to be free'd.
   * @param numTrials [in] integer, number of trials to be free'd.
   * 
   */
  void cperfdmf_freeTrialList(cperfdmf_trial_t* trials, int numTrials); 

  /*@}*/  // end of DB (doxygen) group


  /**
   * @fn int cperfdmf_loadTauOutput(char *path)
   * Load raw TAU output files (profile.*)
   *
   * @param path [in] directory containing profile files
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_NO_PROFILES could not find profile.* files in the given path
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_loadTauOutput(char *path);


  /**
   * @fn int cperfdmf_getMetrics(char*** metrics, int* numMetrics)
   * Get the metrics for the current session
   *
   * @param metrics [out] pointer to an array of c-strings.
   *    The array will be allocated and generated by the API.  The user may free it at any time.
   * @param numMetrics [out] pointer to an integer.  The value will contain the 
   *    number of metrics (the size of the metrics array)
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getMetrics(char*** metrics, int* numMetrics);


  /**
   * @fn int cperfdmf_getFunctions(char*** functions, int* numFunctions)
   * Get the list of functions for the current session
   *
   * @param functions [out] pointer to an array of c-style strings.
   *    The array will be allocated and generated by the API.  The user may free it at any time.
   * @param numFunctions [out] pointer to an integer.  The value will contain the 
   *    number of functions (the size of the functions array)
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getFunctions(char*** functions, int* numFunctions);


  /**
   * @fn int cperfdmf_getNumNodes(int* numNodes)
   * Get the number of nodes for the current session
   *
   * @param numNodes [out] pointer to an integer.  The value will contain the number of nodes for
   *    the current session
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getNumNodes(int* numNodes);


  /**
   * @fn int cperfdmf_getNumContexts(int* numContexts, int node)
   * Get the number of contexts for the given node in the current session
   *
   * @param numContexts [out] pointer to an integer.  The value will contain the number of contexts for
   *    the given node
   * @param node [in] integer, the node to query
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getNumContexts(int* numContexts, int node);


  /**
   * @fn int cperfdmf_getNumThreads(int* numThreads, int node, int context)
   * Get the number of threads for the given node and context in the current session
   *
   * @param numThreads [out] pointer to an integer.  The value will contain the number of threads for
   *    the given node and context
   * @param node [in] integer, the node to query
   * @param context [in] integer, the context to query
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getNumThreads(int* numThreads, int node, int context);


  /**
   * @fn int cperfdmf_getMeanValues(cperfdmf_profile_t *prof, int functionId, int metric)
   * Get mean profile values for the current session
   *
   * @param prof [out] pointer to a cperfdmf_profile_t.  prof will contain the profile retrieved.
   * @param functionId [in] which profile should be retrieved.  The ascii name for a given index can be found by indexing into the array from cperfdmf_getFunctions.
   * @param metric [in] which metric to retrieve (likewise indexed from cperfdmf_getMetrics
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_NON_EXISTENT No mean values were found (this probably indicates something is wrong with the data)
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getMeanValues(cperfdmf_profile_t *prof, int functionId, int metric);

  /**
   * @fn int cperfdmf_getTotalValues(cperfdmf_profile_t *prof, int functionId, int metric)
   * Get total profile values for the current session
   *
   * @param prof [out] pointer to a cperfdmf_profile_t.  prof will contain the profile retrieved.
   * @param functionId [in] which profile should be retrieved.  The ascii name for a given index can be found by indexing into the array from cperfdmf_getFunctions.
   * @param metric [in] which metric to retrieve (likewise indexed from cperfdmf_getMetrics
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_NON_EXISTENT No total values were found (this probably indicates something is wrong with the data)
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getTotalValues(cperfdmf_profile_t* prof, int functionId, int metric);

  /**
   * @fn int cperfdmf_getValues(cperfdmf_profile_t *prof, int functionId, int metric, int node, int context, int thread)
   * Get profile values for the current session
   *
   * @param prof [out] pointer to a cperfdmf_profile_t.  prof will contain the profile retrieved.
   * @param functionId [in] which profile should be retrieved.  The ascii name for a given index can be found by indexing into the array from cperfdmf_getFunctions.
   * @param metric [in] which metric to retrieve (likewise indexed from cperfdmf_getMetrics
   * @param node [in] which node to retrieve
   * @param context [in] which context to retrieve
   * @param thread [in] which thread to retrieve
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_NON_EXISTENT the given node:context:thread did not have a value for this function:metric (this is often expected in the case that a given thread does not call all instrumented functions)
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getValues(cperfdmf_profile_t* prof, int functionId, int metric, int node, int context, int thread);




  /**
   * @fn int cperfdmf_queryNodeExists(int* exists, int node)
   * Query if a given node exists in the trial/profiles
   *
   * @param exists [out] 0 if the node does not exist, 1 if it does
   * @param node [in] integer, which node to query.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_queryNodeExists(int* exists, int node);


  /**
   * @fn int cperfdmf_queryContextExists(int* exists, int node, int context)
   * Query if a given context exists in the trial/profiles
   *
   * @param exists [out] 0 if the context does not exist, 1 if it does
   * @param node [in] integer, which node to query.
   * @param context [in] integer, which context to query.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_queryContextExists(int* exists, int node, int context);


  /**
   * @fn int cperfdmf_queryThreadExists(int* exists, int node, int context, int thread)
   * Query if a given context exists in the trial/profiles
   *
   * @param exists [out] 0 if the thread does not exist, 1 if it does
   * @param node [in] integer, which node to query.
   * @param context [in] integer, which context to query.
   * @param thread [in] integer, which thread to query.
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_queryThreadExists(int* exists, int node, int context, int thread);


  /**
   * @fn int cperfdmf_getGroups(char*** groups, int* numGroups)
   * Retrieve the group names
   *
   * @param groups [out] pointer to array of c-style strings.
   *    The array will be allocated and generated by the API.  The user may free it at any time.
   * @param numGroups [out] pointer to an integer.  The value will contain the 
   *    number of groups (the size of the groups array)
   * 
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */  
  int cperfdmf_getGroups(char*** groups, int* numGroups);


  /**
   * @fn int cperfdmf_getUserEvents(cperfdmf_userEvent_t **userEvents, int *numUserEvents, int node, int context, int thread)
   * Retrieve the user events for the current session
   *
   * @param userEvents [out]
   * @param numUserEvents [out]
   * @param node [in]
   * @param context [in]
   * @param thread [in]
   *
   * @retval CPERFDMF_OK ok
   * @retval CPERFDMF_ERR_SESSION_NOT_INITIALIZED session is not initialized
   * @retval CPERFDMF_NON_EXISTENT No user events were found for the requested node:context:thread
   * @retval CPERFDMF_ERR_INTERNAL_ERROR use getExtendedError() for details
   */
  int cperfdmf_getUserEvents(cperfdmf_userEvent_t **userEvents, int *numUserEvents, int node, int context, int thread);


  /**
   * @fn char *cperfdmf_getExtendedError(void)
   * Get extended error information.  
   * Use this function to get extended error information when a function has returned the CPERFDMF_ERR_INTERNAL_ERROR value
   * @return char* : c-string containing the extended error string, do not free().  
   */
  char *cperfdmf_getExtendedError(void);


  /**
   * @fn char *cperfdmf_errToString(int code)
   * Returns a string equivalent of the given code
   * 
   * @param code [in] int, the code
   *
   * @return c-string, the string equivalent of 'code'
   */
  char *cperfdmf_errToString(int code);



#ifdef __cplusplus
}
#endif /* __cplusplus */



#endif /* __CPERFDMF_H__ */
