#include "MasterMind_CCA.h"

namespace mm
{
  namespace ccaimpl
  {

    /** 
     * Default constructor
     */
    MasterMind::MasterMind ()
    {
      init_mport=false;
      init_tree = false;
      frameworkServices = 0;
      measurement_m=0;
      call_path_tree = 0;
    }

    MasterMind::~MasterMind ()
    {
      std::cout<<"Begin MasterMind destructor\n";
      //delete records
      if(records.size() > 0){
	map<string,performance::Record*>::iterator r;
	r=records.begin();
	//delete each record
	while(r!=records.end()){
	  if(r->second){
	    delete r->second;
	  }
	  r++;
	}
	records.clear();

	//delete call path tree
	if(call_path_tree)
	  delete call_path_tree;
      }
      std::cout<<"End MasterMind destructor\n";
    }

    /**
     * This routine can be called to dump the proxy files to disk
     **/
    int MasterMind::go()
    {
      if(records.size() > 0){
        int rank = (*records.begin()->second).getRank();
        map<string,performance::Record*>::iterator r;
        r=records.begin();
        //delete each record, also generate an entry in the
        //database file for each record
        char fname[256];
        string dbfilename = "dbfile";
        sprintf(fname, "%s.%d", dbfilename.c_str(), rank);
        ofstream outfile(fname, std::ios::app);
        outfile<<"# ComponentName \t Filename\n";
        while(r!=records.end()){
          if(r->second){
            //get component_name
	    std::string c_name = (r->first).substr(0,(r->first).find("::"));
            //outfile<<r->first<<"\t"<<(r->second)->getFileName()<<endl;
            outfile<<c_name<<"\t"<<(r->second)->getFileName()<<std::endl;
            r->second->dumpToFile(r->second->getName());
          }
          r++;
        }
	outfile.close();
        string fn = "call_path_tree";
        char ctname[256];
        sprintf(ctname, "%s.%d",fn.c_str(),rank);
        call_path_tree->dumpChildren((string) ctname);
      }
      return 0;
    }
    /**
     * Create a record and store it in the records map
     **/
    void MasterMind::createRecord(string rname)
    {
      //first check to see if we have gotten a measurement port and MPIBorrow port
      //Only do once!!
      if(!init_mport){
	//Get Measurement port
	classic::gov::cca::Port * port=frameworkServices->getPort("MeasurementPort");
	if(port)
	  measurement_m=dynamic_cast < performance::ccaports::Measurement * >(port);
	if(measurement_m==0){
	  std::cerr << "Connected to something other than a Measurement port" << std::endl;
	}

	init_mport=true;
      }//if

      //if no record with that name exists, create it.  Otherwise, do nothing.
      if(records.find(rname) == records.end()){
	performance::Record *r = new performance::Record(rname, frameworkServices, measurement_m);
	//check to see if we are using MPI.  To do this, see if MPI_INIT has been   
	//called.  If it hasn't set rank to 0 for single cpu case.  Else, get   
	//a communicator and determine the cpu's rank.  
	int flag;
	MPI_Initialized(&flag);
	if(!flag){
	  r->setRank(0);
	}
	else{
	  classic::gov::cca::MPIBorrow * pMPI_Port = dynamic_cast< classic::gov::cca::MPIBorrow *> (frameworkServices->getPort("mpi"));
	  //otherwise, get the communicator and set the rank 
	  int key,tag,rank;
	  MPI_Comm comm = pMPI_Port->borrowComm(1, &tag, key);
	  MPI_Comm_rank(comm, &rank);
	  r->setRank(rank);
	}
	r->setFileName(rname);
	records[rname]=r;
      }//if
    }

    /**
     * Start monitoring the given record
     **/    
    void MasterMind::startMonitoring(string rname)
    {
      //if no record with that name exists, create it.  
      if(records.find(rname) == records.end()){
	createRecord(rname);
      }

      //first check to see if we have initialized the call path tree
      //if we haven't, do so, otherwise, we need to advance
     string node_name = rname.substr(0,rname.find("::"));
     if(!init_tree){
	call_path_tree = new CallPathNode(node_name);
	init_tree = true;
      }
      else{
	call_path_tree = call_path_tree->getChild(node_name);
      }
     //(*records[rname]).setParameters(m);
      (*records[rname]).startTiming(true);
    }

    /**
     * Stop monitoring the given record
     **/
    void MasterMind::stopMonitoring(string rname, map<string, double> &params)
    {
      map<string,double> counters;
      (*records[rname]).stopTiming(true, counters, params);
      //when we stop monitoring, assume that compute in component is over, so 
      //we can move back "up" the call-path tree (first, increment the performance 
      //counters for the current node)
      call_path_tree->incrementCounters(counters);
      //before moving back up tree, make sure we have a valid pointer for the root
      //if we don't assume we are at the root.
      if(call_path_tree->getParent())
	call_path_tree = call_path_tree->getParent();
    }

    /**
     * Set dump file name
     **/
    void MasterMind::setFileName(string rname, string fname){
      (*records[rname]).setFileName(fname);
    }

    /**
     * Dump data for the given record
     **/
    void MasterMind::dumpData(string rname)
    {
      (*records[rname]).dumpToFile(rname);
    }

    /**
     * Dump data for the given record to the given file
     **/
    void MasterMind::dumpData(string rname, string fname)
    {
      (*records[rname]).setFileName(fname);
      (*records[rname]).dumpToFile(rname);
    }

    /**
     * Destroy the given record
     **/
    void MasterMind::destroyRecord(string rname)
    {
      map<string,  performance::Record *>::iterator it;
      it=records.find(rname);
      delete it->second;
      records.erase(it);    
    }


    int MasterMind::getPerformanceData(std::string rname, std::vector< std::vector<double> > &data, bool reset)
    {
      //Does the component name we are given exist?
      if(records.find(rname) == records.end()){
	std::cerr<<"Unable to find component record for : '"<<rname<<"'"<<std::endl;
        return -1;
      }
      performance::Record * rec = records.find(rname)->second;
      rec->getParameters(data);
      //do we reset the parameters?
      if(reset){
	rec->resetParameters();
      }
      return 1;
    }

    int MasterMind::getCompMethNames(std::vector<string> &cm_names)
    {
      std::map<std::string, performance::Record*>::iterator it;
      for(it = records.begin(); it != records.end(); it++){
	cm_names.push_back(it->first);
      }
      return 1;
    }

    /**
     * The framework passes a pointer to the Services object
     * via this method, which is called by the framework as soon
     * as the component is instantiated.
     */
    void MasterMind::setServices (classic::gov::cca::Services * services)
    {
      if (services != 0)
        {
          frameworkServices = services;

          // Provide a Monitor port
          classic::gov::cca::PortInfo * portInfo =
            frameworkServices->createPortInfo ("MonitorPort",
                                               "monitor.ccaports.Monitor",
                                               0);
          if (portInfo != 0)
            frameworkServices->addProvidesPort (this, portInfo);

	  // Provide a GoPort
	  portInfo = frameworkServices->createPortInfo("Go", "gov.cca.GoPort",0);
	  if(portInfo != 0)
	    frameworkServices->addProvidesPort(this, portInfo);

	  //Provide a PerfParamPort
          char * props[3];
          props[0] = "SupportsKind";
          props[1] = "c++-cca-spec-neo";
          props[2] = 0;
	  portInfo = frameworkServices->createPortInfo ("PerfParamPort",
							"perfparam.ccaports.PerfParam",
							props);
	  if (portInfo != 0)
            frameworkServices->addProvidesPort (this, portInfo);

	  // Use a Measurement port
	  portInfo = 
	    frameworkServices->createPortInfo ("MeasurementPort",
					       "performance.ccaports.Measurement",
					       0);
	  frameworkServices->registerUsesPort(portInfo);
	
	  // Use a MPIBorrow port
	  frameworkServices->registerUsesPort( frameworkServices->createPortInfo("mpi", "gov.cca.MPIBorrow",0));
        }
      else
        {
          // Close down if not closed already
          if (frameworkServices){
     		frameworkServices->releasePort ("MeasurementPort");
     		frameworkServices->releasePort("mpi");
          	frameworkServices->removeProvidesPort ("MonitorPort");
		frameworkServices->removeProvidesPort ("PerfParamPort");
		frameworkServices->removeProvidesPort("Go");
	  }
	    frameworkServices = 0;
        }
    }

  }
}

extern "C"
{

  // We need a C wrapper to get dynamic loading to work.
  // Here's our NULL constructor wrapped in C.
  classic::gov::cca::Component * create_MasterMind ()
  {
    classic::gov::cca::Component * wrapper;
    mm::ccaimpl::MasterMind * component;
    component = new mm::ccaimpl::MasterMind ();
    wrapper = dynamic_cast < classic::gov::cca::Component * >(component);
    return wrapper;
  }

  // See dccafe/cxx/dc/framework/ComponentFactory.h for details.
  // More sophisticated repository interfaces would supplant this.

  char **getComponentList ()
  {
    static char *list[2];
    list[0] = "create_MasterMind MasterMind";
    list[1] = 0;
    return list;
  }
};
