#ifndef MEASUREMENT_CCA_H
#define MEASUREMENT_CCA_H

#include <string>
using std::string; 

namespace performance
{
  class Timer 
  {
    public: 

      /**
       * The destructor should be declared virtual in an interface class.
       */
      virtual ~Timer() 
      {
      }

      /**
       * Start the Timer.
       * Implement this function in
       * a derived class to provide required functionality.
       */ 
      virtual void start(void) = 0; 

      /**
       * Stop the Timer.
       */ 
      virtual void stop(void) = 0;

      /**
       * Set the name of the Timer.
       */ 
      virtual void setName(string name) = 0;

      /**
       * Get the name of the Timer.
       */ 
      virtual string getName(void) = 0;

      /**
       * Set the type information of the Timer
       * (e.g., signature of the routine) 
       */ 
      virtual void setType(string name) = 0;

      /**
       * Get the type information of the Timer
       * (e.g., signature of the routine) 
       */ 
      virtual string getType(void) = 0;

      /**
       * Set the group name associated with the Timer
       * (e.g., All MPI calls can be grouped into an "MPI" group)
       */ 
      virtual void setGroupName(string name) = 0;

      /**
       * Get the group name associated with the Timer
       */ 
      virtual string getGroupName(void) = 0;

      /**
       * Set the group id associated with the Timer
       */ 
      virtual void setGroupId(unsigned long group ) = 0;

      /**
       * Get the group id associated with the Timer
       */ 
      virtual unsigned long getGroupId(void) = 0;

  };

  /**
   * Query the timing information
   */ 
  class Query
  {
    public:
      /**
       * Destructor
       */ 
      virtual ~Query() 
      {
      }

      /**
       * Get the list of Timer names
       */ 
      virtual void getTimerNames(const char **& functionList, int& numFuncs) = 0;
      /**
       * Get the list of Counter names
       */ 
      virtual void getCounterNames(const char **& counterList, 
		      int& numCounters) = 0;

      /** 
       * getTimerData. Returns lists of metrics.
       */
      virtual void getTimerData(const char **& inTimerList, 
		    int numTimers, double **& counterExclusive, 
		    double **& counterInclusive, int*& numCalls, 
		    int*& numChildCalls, const char **& counterNames, 
		    int& numCounters) = 0;

      /** 
       * dumpProfileData. Writes the entire profile to disk in a dump file. 
       * It maintains a consistent state and represents the instantaneous 
       * profile data had the application terminated at the instance this call
       * is invoked.
       */
      virtual void dumpProfileData(void) = 0;
  
      /** 
       * dumpProfileDataIncremental. Writes the entire profile to disk in a 
       * dump file whose name contains the current timestamp. 
       * It maintains a consistent state and represents the instantaneous 
       * profile data had the application terminated at the instance this call
       * is invoked. This call allows us to build a set of timestamped profile
       * files. 
       */
      virtual void dumpProfileDataIncremental(void) = 0;
  
      /** 
       * dumpTimerNames. Writes the list of timer names to a dump file on the 
       * disk.
       */
      virtual void dumpTimerNames(void) = 0;
  
      /** 
       * dumpTimerData. Writes the profile of the given set of timers to the 
       * disk. This allows the user to select the set of routines to dump and
       * periodically write the performance data of a subset of timers to disk
       * for monitoring purposes.
       */
      virtual void dumpTimerData(const char **& inTimerList, int numTimers) = 0;
  
      /** 
       * dumpTimerDataIncremental. Writes the profile of the given set of 
       * timers to the disk. The dump file name contains the current timestamp 
       * when the data was dumped. This allows the user to select the set of 
       * routines to dump and periodically write the performance data of a 
       * subset of timers to the disk and maintain a timestamped set of values 
       * for post-mortem analysis of how the performance data varied for a 
       * given set of routimes with time.
       */
      virtual void dumpTimerDataIncremental(const char **& inTimerList, 
		      int numTimers) = 0;
  };

  /**
   * User defined event profiles application specific events
   */ 
  class Event
  {
    public:
      /**
       * Destructor
       */ 
      virtual ~Event() 
      {
      }

      /**
       * Register the name of the event
       */ 
      virtual void trigger(double data) = 0;
  };

  /**
   * This class implements the runtime instrumentation control based on groups
   */
  class Control
  {
    public:
      /**
       * Destructor
       */ 
     virtual ~Control ()
      {
      }

      /** 
       * Control instrumentation. Enable group Id.
       */
      virtual void enableGroupId(unsigned long id) = 0;
      
      /** 
       * Control instrumentation. Disable group Id.
       */
      virtual void disableGroupId(unsigned long id) = 0;
  
      /** 
       * Control instrumentation. Enable group name.
       */
      virtual void enableGroupName(string name) = 0;
      
      /** 
       * Control instrumentation. Disable group name.
       */
      virtual void disableGroupName(string name) = 0;
  
      /** 
       * Control instrumentation. Enable all groups.
       */
      virtual void enableAllGroups(void) = 0;
  
      /** 
       * Control instrumentation. Disable all groups.
       */
      virtual void disableAllGroups(void) = 0;
    };

  namespace ccaports
  {
    /**
     * This abstract class declares the Measurement interface.  
     * Inherit from this class to provide functionality.
     */
    class Measurement:
      public virtual classic::gov::cca::Port
    {
    public:

      /**
       * The destructor should be declared virtual in an interface class.
       */
      virtual ~ Measurement ()
      {
      }

      /**
       * Create a Timer
       */ 
      virtual performance::Timer* createTimer(void) = 0; 
      virtual performance::Timer* createTimer(string name) = 0; 
      virtual performance::Timer* createTimer(string name, string type) = 0; 
      virtual performance::Timer* createTimer(string name, string type, 
		      string group) = 0; 

      /**
       * Create a Query interface
       */ 
      virtual performance::Query* createQuery(void) = 0; 

      /**
       * Create a User Defined Event interface
       */ 
      virtual performance::Event* createEvent(void) = 0; 
      virtual performance::Event* createEvent(string name) = 0; 

      /**
       * Create a Control interface for selectively enabling and disabling
       * the instrumentation based on groups 
       */ 
      virtual performance::Control* createControl(void) = 0; 
    };

  }
}
#endif                          // MEASUREMENT_CCA_H
