#include "Record_CCA.h"

namespace performance{
  //constructor, initialize variables to zero
  Record::Record(string n,classic::gov::cca::Services * fs, performance::ccaports::Measurement * m){
    svc = fs;
    name = n;
    measurement_m=m;
    timer = 0;
    pnames_set=false;
    filename_exists = false;
  }

  Record::~Record(){
    //dumpToFile(name);
  }

  //routines
  string Record::getName(){
    return name;
  }

  int Record::setRank(int r){
    rank = r;
    return 1;
  }

  int Record::getRank(){
    return rank;
  }

  void Record::getParameters(std::vector<std::vector<double> > &p)
  {
    p=parameters;
  }
  
  void Record::resetParameters()
  {
    parameters.clear();
  }

  int Record::startTimer(){
    //check to see if we have created a timer
    if(timer == 0)
      timer = measurement_m->createTimer(name);
    timer->start();
    return 1;
  }
  
  int Record::stopTimer(){
    timer->stop();
    return 1;
  }

  int Record::startMPITimers(){
    Control* tc = measurement_m->createControl();
    tc->enableGroupId(MPI_ID);
    return 1;
  }
  int Record::stopMPITimers(){
    Control* tc = measurement_m->createControl();
    tc->disableGroupId(MPI_ID);
    return 1;
  }

  int Record::startTiming(bool mpitimers){
    //if mpitimers are set, stop mpi timing
    if(mpitimers)
	    stopMPITimers();

    //query the TAU library and get all the timing/counter data
    Query* q = measurement_m->createQuery();
    const char **funcList, **counterNames;
    int numFuncs, numCounters;
    double **counterExclusive, **counterInclusive;
    int *numCalls, *numChildCalls;
    q->getTimerNames(funcList,numFuncs);
    q->getCounterNames(counterNames,numCounters);
    q->getTimerData(funcList,numFuncs,counterExclusive,counterInclusive,
    numCalls,numChildCalls,counterNames,numCounters);
    
    int i,j;
    //first, find the data for the current function.  Then update the 
    //last_data_* map to contain the current values for the counters.
    for(j = 0; j < numFuncs; j++){
      if((string)funcList[j] == name){
        for(i = 0; i < numCounters; i++){
          last_data_exclusive[(string)counterNames[i]] = counterExclusive[j][i];
	  last_data_inclusive[(string)counterNames[i]] = counterInclusive[j][i];
        }
      }
      //if mpitimers are set, determine current mpi time.
      if(mpitimers){
        if(((string)funcList[j]).substr(0,4) == "MPI_")
  	  for(i = 0; i < numCounters; i++){
		mpi_last_data[(string)counterNames[i]] += counterExclusive[j][i];
     	  }
      }
    }//for
    
    if(mpitimers)
      startMPITimers();
    startTimer();
    return 1;
  }

  int Record::stopTiming(bool mpitimers, map<string,double> &pc, map<string,double> &params){
    stopTimer();
    if(mpitimers)
      stopMPITimers();

    //First, let's handle the parameters
    //create a map to hold the input parameters and timing data
    //for this call.  Set it to the parameters map to start.
    map<string, double> m = params;
    //add the parameters and variable names for the parameters
    map<string, double>::iterator it;
    vector<double> v;
    for(it = params.begin(); it != params.end(); it++){
      v.push_back(it->second);
    }

    parameters.push_back(v);

    //Have set the variable names?
    if(!pnames_set){
      for(it = params.begin(); it != params.end(); it++){
	param_names.push_back(it->first);
      }
      pnames_set = true;
    }

    Query* q = measurement_m->createQuery();
    const char **funcList, **counterNames;        
    int numFuncs, numCounters;
    double **counterExclusive, **counterInclusive;
    int *numCalls, *numChildCalls;
    q->getTimerNames(funcList,numFuncs);
    q->getCounterNames(counterNames,numCounters);
    q->getTimerData(funcList,numFuncs,counterExclusive,counterInclusive,
		    numCalls,numChildCalls,counterNames,numCounters);
    int i,j;
      
    //Cycle through the functions and determine the values for the last timing session.
    string excl = "Exclusive ";
    string incl = "Inclusive ";
    for(i=0;i<numFuncs;i++){
      if((string)funcList[i] == name){
	//check to see if we have timing data
	for(j = 0; j < numCounters; j++){
	  //get the countername
	  string countername = (string) counterNames[j];
	  string newname = countername;
	  //if the counter name is "default counter", it is really wall-clock time
	  //so change it to "Time" for a clearer keyword
	  if(countername == "default counter")
	    newname = "Compute_Time";
	  //insert the pair into the map with parameters
	  m.insert(make_pair(newname, counterExclusive[i][j] - last_data_exclusive[countername]));
	  //now, insert the inclusive and exlusive counters into the map
	  //reference used by the call path tree
	  pc.insert(make_pair(excl + newname, counterExclusive[i][j] - last_data_exclusive[countername]));
	  pc.insert(make_pair(incl + newname, counterInclusive[i][j] - last_data_inclusive[countername]));
	 }
       }	
  
       //if mpitimers are set, determine current mpi time.
       if(mpitimers){
         if(((string)funcList[i]).substr(0,4) == "MPI_")
           for(j = 0; j < numCounters; j++){
             mpi_stop_data[(string)counterNames[j]] += counterExclusive[i][j];
           }
       }
    }//for

    //Enter in the MPI counts into the map
    map<string,double>::iterator a = mpi_last_data.begin();
    while(a != mpi_last_data.end()){
      string s;
      if(a->first == "default counter")
	s = "MPI_Time";
      else
	s = "MPI_Time_" + a->first;
      m.insert(make_pair(s, mpi_stop_data[(string)a->first] - a->second));
      a++;
    }//while

    //now, clear the the last data maps
    mpi_last_data.clear();
    mpi_stop_data.clear();
    last_data_exclusive.clear();
    last_data_inclusive.clear();

    //add the map to the deque
    t_data.push_back(m);
    return 1;
  }
  
  string Record::getFileName(){
    return filename;
  }

  int Record::setFileName(string s){
    // can only set filename once return an error if filename is already set
    if(filename_exists)
      return -1;
    char fname[256];
    sprintf(fname,"%s.%d", s.c_str(), getRank());
    filename = (string) fname;
    filename_exists = true;
    return 1;
  }
  
  int Record::dumpToFile(string rname){ 
    //check to see if we have a filename
    if(filename == ""){
      cerr<<"No dump filename for the given record"<<endl;
      return -1;
    }
    ofstream outfile(filename.c_str(), std::ios::app);  
    outfile<<"# "<<rname<<endl;
    outfile<<"#\t";
    map<string,double> m = t_data.front();
    map<string,double>::iterator it = m.begin();
    //The first line should have MPI_Time and Compute_Time,
    //as well as any parameters we tracked.
    //We want MPI_Time and Compute_Time to be listed first
    outfile<<"Compute_Time\tMPI_Time\t";

    while(it!=m.end()){
      if(it->first != "Compute_Time" && it->first != "MPI_Time")
	outfile<<it->first<<"\t";
      it++;
    }
    //now, dump the data.  But, remember, we need to output
    //Compute_Time and then MPI_Time, first.
    outfile<<endl;
    while(!t_data.empty()){
      m = t_data.front();
      t_data.pop_front();
      it = m.find("Compute_Time");
      outfile<<"\t"<<it->second;
      m.erase(it);
      it = m.find("MPI_Time");
      outfile<<"\t"<<it->second;
      m.erase(it);
      it = m.begin();
      while(it!=m.end()){
	outfile<<"\t"<<it->second;
	it++;
      }//while
      outfile<<endl;
    }
    outfile.close();
    return 1;
  } 
}
