/******************************************************************************
 * Nick Trebon
 *
 * 11/12/2003
 *
 * This program was created to take a call path tree generated by dump of 
 * a CallPathNode object and perform some pruning operations on it.
 *
 *****************************************************************************/
#define USE_AVERAGE 1

#include "CallPathNode.h"
#include <fstream>
#include <iostream>
#include <map>
#include <vector>
#include <string>
#include <getopt.h>

using namespace std;

ifstream infile;

void createNode(CallPathNode * tree, string n){
  //First, parse the next line of data from input_file
  if(!infile.eof()){
    string name, significant, count, counters,parent, children;
    getline(infile,name, '#');
    getline(infile,significant, '#');
    getline(infile,count, '#');
    getline(infile,counters, '#');
    getline(infile,parent, '#');
    getline(infile,children);
    //Are we on the right node?
    if(n != name){
      cout<<"Oops, we are trying to build node with name: "<<n<<", but we read in name: "<<name<<endl;
      exit(0);
    }
    tree = tree->newChild(name);
   //Fill in the rest of the nodes attributes                                       
    int c = atoi(count.c_str());
    tree->setCount(c);
    //Now create map of counters                                 
    //First, cycle through counters string until we have only one item left                
    string cname;
    double cvalue;
    map<string,double> m;
    while(counters.find("\\") != string::npos){
      cname = counters.substr(0,counters.find("="));
      counters = counters.substr(counters.find("=")+1);
      cvalue = atof(counters.substr(0, counters.find("\\")).c_str());
      counters = counters.substr(counters.find("\\")+1);
      m.insert(make_pair(cname,cvalue));
    }
    //Now, handle the final (or single) counter case                                     
    cname = counters.substr(0,counters.find("="));
    cvalue = atof(counters.substr(counters.find("=")+1).c_str());
    m.insert(make_pair(cname,cvalue));
    tree->setCounters(m);  
    
    //Handle the children
    if(children != ""){
      while(children.find("\\") != string::npos){
        string temp_string = children.substr(0,children.find("\\"));
        children = children.substr(children.find("\\")+1);
        createNode(tree,temp_string);
      }//while                                                                      
      //process the last(only) child                                                
      createNode(tree,children);
    }//if  
    
    //when all the children are processed, set the current tree pointer
    //to that of its parent (unless we are at the root)
    if(tree->getParent())
      tree = tree->getParent();
  }
  else{
    cout<<"Oops.  We have a file IO error!"<<endl;
    exit(0);
  }//else
}

CallPathNode * generateTree(string filename, CallPathNode * tree){
  //First open up the input file
  infile.open(filename.c_str(), ios::in );
  //check for problem
  if(infile.bad()){
    cerr<<"Unable to open file: "<<filename<<endl;
    exit(0);
  }

  //Create the root, then make the rest of the tree using a recursive subroutine
  if(!infile.eof()){
    string name, significant, count, counters,parent, children;
    getline(infile,name, '#');
    getline(infile,significant, '#');
    getline(infile,count, '#');
    getline(infile,counters, '#');
    getline(infile,parent, '#');
    getline(infile,children);
    tree = new CallPathNode(name);
    //Fill in the rest of the nodes attributes
    int c = atoi(count.c_str());
    tree->setCount(c);
    //Now create map of counters
    //First, cycle through counters string until we have only one item left
    string cname;
    double cvalue;
    map<string,double> m;       
    while(counters.find("\\") != string::npos){
      cname = counters.substr(0,counters.find("="));
      counters = counters.substr(counters.find("=")+1);
      cvalue = atof(counters.substr(0, counters.find("\\")).c_str());
      counters = counters.substr(counters.find("\\")+1);
      m.insert(make_pair(cname,cvalue));
    }
    //Now, handle the final (or single) counter case
    cname = counters.substr(0,counters.find("="));
    cvalue = atof(counters.substr(counters.find("=")+1).c_str());
    m.insert(make_pair(cname,cvalue));
    tree->setCounters(m);
    
    //Handle the children
    //for each child from the root, call the recursive routine createNode()
    if(children != ""){
      while(children.find("\\") != string::npos){
	string temp_string = children.substr(0,children.find("\\"));
	children = children.substr(children.find("\\")+1);
	createNode(tree, temp_string);
      }//while
      //process the last(only) child
      createNode(tree,children);
    }//if
  }//if
  infile.close();
  return tree;
}

void pruneTree(double prune_level, double significance_level, CallPathNode* tree){
  //Get the inclusive count for node's children
  double total = tree->getChildrenInclusiveCounter();
  //The first test is to see if the children contribute a significant portion of the compute time, 
  //or if the majority of work is done at the current node
  if(total < (significance_level * tree->getInclusiveCounter())){
    cout<<"We have a insignificance case!!!"<<endl;
    //The children of the current node are insignificant, so prune them off
    vector<CallPathNode *> v = tree->getChildren();
    vector<CallPathNode *>::iterator vit;
    for(vit = v.begin(); vit != v.end(); vit++){
      tree->removeChild(*vit);
    }
    //now, there is nothing else to do from this node
    return;
  }
  //Check to see if the current node is insignificant in relation
  //to its children ie, are the children responsible for most of
  //the current node's inclusive time?
  else if(total > ((1-significance_level) * tree->getInclusiveCounter())){
    cout<<"Setting node as insignificant at node: "<<tree->getName()<<endl;
    tree->setSignificant(false);
  }

#ifdef USE_AVERAGE
  int num_children = tree->getNumChildren(); 
  double average = total / num_children;
#endif
  vector<CallPathNode *> children = tree->getChildren();
  vector<CallPathNode *>::iterator iter = children.begin();
  //iterate through the children and check to see if we should prune any off.  The
  //children we don't prune, pass as the tree into a recursive call of this routine.
  while(iter != children.end()){
    //get the inclusive count for the child
    double counter = (*iter)->getInclusiveCounter();
    //check to see if we should prune
#ifdef USE_AVERAGE
    cout<<"Using average for pruning level"<<endl;
    if(counter < (prune_level * average)){
#else
    cout<<"Using total for purning level"<<endl;
    if(counter < (prune_level * total)){
#endif
      cout<<"Pruning child branch rooted at: "<<(*iter)->getName()<<endl;
      tree->removeChild(*iter);
      children.erase(iter);
    }//if
    else{
      //else, recursively call this routine on children that are NOT leaf nodes
      if( (*iter)->getNumChildren() > 0)
	pruneTree(prune_level, significance_level, (*iter));
      //also, increment iterator here
      if(iter != children.end())
	iter++;
    }//else
  }//while
}//pruneTree()


int main(int argc, char ** argv){
  int ch; //used in switch statement for parsing command line arguments
  string dumpfilename = "call_path_tree.0";
  int help = 0;
  double prune_level = .1;
  double significance_level = .1;

  /* -- parse command line arguments ---------------------------------------- */
  while ( (ch = getopt (argc, argv, "f:p:s:h")) != EOF ) {
    switch ( ch ) {
    case 'f': /* -- name of CallPathNode dump *F*ile to read in -- */
      dumpfilename = optarg;
      break;
    case 'h': /* -- help - print out a usage and exit -- */
      help = 1;
      break;
    case 'p':  /* -- *P*runing level -- */
      prune_level = atof(optarg);
      break;
    case 's':  /* -- *S*ignificance level -- */
      significance_level = atof(optarg);
      break;
    }//switch
  }//while

  if(help){
    cout<<"usage: [-f <filename>] [-h] [-p <p_level>] [-s <s_level>]"<<endl;
    cout<<" -f filename : specify filename of dump file used to generate tree (default: call_path_tree.0)"<<endl;
    cout<<" -p p_level: set prune level.  Value should be between 0 and 1, and represents the percentage of inclusive count a branch must contribute to the combined inclusive count of itself and all its siblings for the branch to be kept"<<endl;
    cout<<" -s s_level: set children significance level.  Value should be between 0 and 1.  If the combined inclusive count for a node's children is less than the product of this value and the parent node's inclusive count, prune all children as they do not contribute a significant amount of computation"<<endl;
    cout<<" -h : help - print out this message and exit"<<endl;
    exit(1);
  }

  CallPathNode * tree = 0;
  //first we need to generate the tree from the dumpfile
  tree = generateTree(dumpfilename, tree);
  tree->dumpChildren("tree.out");

  tree->getRootPointer();
  
  pruneTree(prune_level, significance_level, tree);
  tree->dumpChildren("pruned.out");

  if(tree)
    delete tree;
  return 0;
}
