#include "PerfModelLib.h"
#include <fstream>
#include <iostream>
#include <stdlib.h>

std::ifstream& getFileHandle(){
  static std::ifstream infile;
  return infile;
}


namespace modeling{
  Evaluator::Evaluator(){
    mintree = 0;
  }
  
  Evaluator::~Evaluator(){
    if(mintree)
      delete mintree;
  }

  int Evaluator::setDBFile(std::string filename)
  {
    //open the file
    std::ifstream infile;
    infile.open(filename.c_str(), std::ios::in);
    //check for a file error
    if(infile.bad())
      return -1;
    //cycle through the lines of input file and set up database
    std::string cname, location;
    while(!infile.eof() ){
      if(infile.peek() != '#'){
	getline(infile, cname, '\t');
	getline(infile, location);
	//only continue if there is something in cname
	if(cname.size() > 0){
	  //does the component name already exist in the database?
	  //if so, we need to add the location to the vector,
	  //otherwise, create a new entry
	  std::map<std::string, std::vector<std::string> >::iterator it = database.find(cname);
	  if(it != database.end()){
	    it->second.push_back(location);
	  }
	  else{
	    //create a new entry
	    std::vector<std::string> v;
	    v.push_back(location);
	    database.insert(make_pair(cname, v));
	  }
	}
      }
      else{
	//if we get here, the line must be a comment, so ignore
	infile.ignore(256,'\n');
      }
    }
    //close up file
    infile.close();
  
    //return success
    return 1;
  }

  void Evaluator::printAllDBEntries(){
    for(std::map<std::string,std::vector<std::string> >::iterator it = database.begin(); it != database.end(); it++){
      std::cout<<it->first<<std::endl;
      for(std::vector<std::string>::iterator vit = it->second.begin(); vit != it->second.end(); vit++){
	std::cout<<"\t"<<*vit<<std::endl;
      }//for
    }//for
  }

  int Evaluator::setFamily(std::string family_name, std::vector<AbstractFamilyMember *>f_members)
  {
    //check to make sure that each family member only belongs to one family
    std::vector<AbstractFamilyMember *>::iterator vit;
    std::vector<AbstractFamilyMember *>::iterator it;
    std::map<std::string,std::vector<AbstractFamilyMember *> >::iterator mit;
    for(vit = f_members.begin(); vit != f_members.end(); vit++){
      for(mit = families.begin(); mit != families.end(); mit++){
	for(it = mit->second.begin(); it != mit->second.end(); it++){
	  if((*it)->getName() == (*vit)->getName()){
	    //we have found a previous occurence of the member
	    //keep the first occurence, and delete the newest entry
	    std::cerr<<family_name<<": We have already entered the family member: "<<(*vit)->getName()<<" into family: "<<mit->first<<std::endl;
	    vit = f_members.erase(vit);
	    vit = vit - 1;
	    //to speed up the process -- there are no duplicates, so if we find that that family member we 
	    //are trying to add already exists, we can stop after we find the first(only) occurence.
	    //and, if f_members.size() is 0 after we remove the duplicate, then don't put in an empty family
	    //Report a warning that we are removing an empty family
	    if(f_members.size() == 0){
	      std::cerr<<"WARNING: attempt to add an empty family.  The family '"<<family_name<<"' will not be added."<<std::endl;
	      return 1;
	    }
	    //if there are still family_members to add, we know we can stop searching for the one we just deleted, so
	    //exit the inner-most for loop.
	    it = (mit->second.end() - 1);
	  }//if
	}//for
      }//for
    }//for
    families.insert(make_pair(family_name, f_members));
    return 1;
  }

  void Evaluator::printFamilies(){
    std::map<std::string, std::vector<AbstractFamilyMember *> >::iterator mit;
    std::vector<AbstractFamilyMember *>::iterator vit;
    for(mit = families.begin(); mit != families.end(); mit++){
      std::cout<<mit->first<<": ";
      for(vit = mit->second.begin(); vit != mit->second.end(); vit++){
	std::cout<<(*vit)->getName()<<" ";
      }//for
      std::cout<<std::endl;
    }//for
  }

  int Evaluator::getNumFamilies()
  {
    return families.size();
  }

  //recursive routine used in constructing minimal call tree
  void recCreateNode(MinTreeNode * tree, std::string n){
    //First, parse the next line of data from input_file
    if(!getFileHandle().eof()){
      std::string name, significant, count, counters,parent, children;
      getline(getFileHandle(),name, '#');
      getline(getFileHandle(),significant, '#');
      getline(getFileHandle(),count, '#');
      getline(getFileHandle(),counters, '#');
      getline(getFileHandle(),parent, '#');
      getline(getFileHandle(),children);
      //Are we on the right node?
      if(n != name){
	std::cerr<<"Oops, we are trying to build node with name: "<<n<<", but we read in name: "<<name<<std::endl;
	exit(0);
      }

      /*      //if the name contains '_proxy', remove it
      if(name.find("_proxy")){
        name = name.substr(0,name.find("_proxy"));
	}*/

      tree = tree->newChild(name);

      //Handle the children
      if(children != ""){
	while(children.find("\\") != std::string::npos){
	  std::string temp_string = children.substr(0,children.find("\\"));
	  children = children.substr(children.find("\\")+1);
	  recCreateNode(tree, temp_string);
	}//while                                                                      
	//process the last(only) child                                                
	recCreateNode(tree,children);
      }//if  
    
      //when all the children are processed, set the current tree pointer
      //to that of its parent (unless we are at the root)
      if(tree->getParent())
	tree = tree->getParent();
    }
    else{
      std::cerr<<"Oops.  We have a file I/O error!"<<std::endl;
      exit(0);
    }//else
  }

  int Evaluator::setMinimalNetwork(std::string filename)
  {
    //open the file
    getFileHandle().open(filename.c_str(), std::ios::in);
    //check for a file error                                                                                          
    if(getFileHandle().bad()){
      std::cerr<<"Unable to open file: "<<filename<<std::endl;
      return -1;
    }
    //Create the root, then make the rest of the tree using a recursive subroutine
    //Assume that input file is in the form
    //generated by the Mastermind output.  Specifically, that each line represents a
    //node in the following format:
    //#<Node_name>#<Significant#<count>#<counter_name counter_value>\ ...#<parent>#<children_names> \ ...#
    //We, however, are only interested in the name, parent, and children names.
    if(!getFileHandle().eof()){
      std::string name, significant, count, counters,parent, children;
      getline(getFileHandle(),name, '#');
      getline(getFileHandle(),significant, '#');
      getline(getFileHandle(),count, '#');
      getline(getFileHandle(),counters, '#');
      getline(getFileHandle(),parent, '#');
      getline(getFileHandle(),children);
      /*     //if the name contains '_proxy', remove it
      if(name.find("_proxy")){
	name = name.substr(0,name.find("_proxy"));
	}*/
      mintree = new MinTreeNode(name);
 
       //Handle the children
      //for each child from the root, call the recursive routine createNode()
      if(children != ""){
	while(children.find("\\") != std::string::npos){
	  std::string temp_string = children.substr(0,children.find("\\"));
	  children = children.substr(children.find("\\")+1);
	  recCreateNode(mintree, temp_string);
	}//while
	//process the last(only) child
	recCreateNode(mintree, children);
      }//if
    }//if

    //close up file
    getFileHandle().close();

    //return success
    return 1;
  }
 
  int Evaluator::recTransformChildren(MinTreeNode * tree){
   //change the name of the current node to the family name, and add the family name to f_names
    int found = 0;
    std::map<std::string, std::vector<AbstractFamilyMember *> >::iterator mit;
    std::vector<AbstractFamilyMember *>::iterator vit;
    for(mit = families.begin(); mit != families.end(); mit++){
      for(vit = mit->second.begin(); vit != mit->second.end(); vit++){
	if(tree->getName() == (*vit)->getName()){
	  //we have found the component, now change node name to family name and
	  //add family name to vector.  Then update the iterators so we exit the
	  //for loops since we have found the family member.
	  tree->changeName(mit->first);
	  min_families.push_back(mit->first);
	  vit = (mit->second.end() - 1);
	  //mit = families.end();
	  found = 1;
	}
      }
    }
    //if we didn't found, report an error as we have a minimal component that we
    //don't know what family it belongs to (and no performance model has been supplied).
    if(!found){
      std::cerr<<"ERROR:  The core component '"<<tree->getName()<<"' was not found in any family.  Either no performance model has been supplied, or their is a misspelling.  Please check and try again."<<std::endl;
    }

    //now, make the call recursively on the children
    std::vector<MinTreeNode *> children = tree->getChildren();
    std::vector<MinTreeNode *>::iterator it;
    for(it = children.begin(); it != children.end(); it++){
      recTransformChildren(*it);
    }
    return 1;
  }


  int Evaluator::createFamilyCoreTree(){
    //first, check to make sure we have a mintree instance
    if(mintree == 0){
      std::cerr<<"Minimal network is uninitialized.  Please call setMinimalNetwork(filename)"<<std::endl;
      return -1;
    }

    mintree->getRootPointer();

    //Now, search through the tree, and for each node, find which family the component belongs to,
    //and change the name to the family name.  Add that family name into the vector to keep track of which
    //family names appear in the tree.  
    
    //we can skip over the root of the tree, which is the driver.
    
    std::vector<MinTreeNode *> children = mintree->getChildren();
    std::vector<MinTreeNode *>::iterator vit;
    for(vit = children.begin(); vit != children.end(); vit++){
      recTransformChildren(*vit);
    }
    
    return 1;
  }
    
  int Evaluator::printMinFamilies(){
    std::vector<std::string>::iterator it;
    for(it = min_families.begin(); it != min_families.end(); it++){
      std::cout<<*it<<std::endl;
    }
    return 1;
  }

  int Evaluator::recSelectMember(std::vector<AbstractFamilyMember * > &best_set, 
				 std::map<std::string, double> &opt_result, 
				 std::vector<AbstractFamilyMember * > & cset, 
				 std::map<std::string, std::vector<AbstractFamilyMember *> >::iterator family_iter){
    if(family_iter != families.end()){
      std::vector<AbstractFamilyMember *>::iterator vit;
      for(vit = family_iter->second.begin(); vit != family_iter->second.end(); vit++){
	//select the current member, and make a recursive call for the next family
	cset.push_back(*vit);
	recSelectMember(best_set, opt_result, cset, ++family_iter);
	//now, we need to pop the element we chose to move to the next one
	cset.pop_back();
	--family_iter;
      }
    }
    else{
      std::cout<<"We have a complete subset."<<std::endl;
      //if we have family_iter == families.end(), then we know we have one 
      //element from each of the families.  Now, we need to calculate
      //the performance prediction.
      std::vector<AbstractFamilyMember *>::iterator it;
      std::map<std::string, std::vector<std::string> >::iterator mit;
      //we need a map to store the cumulative performance predictions for the current       
      //component instance                                          
      std::map<std::string, double> predictions;
      std::map<std::string, double> instance_predictions;
      for(it = cset.begin(); it != cset.end(); it++){
	mit = database.find((*it)->getName());
	if(mit != database.end()){
	  //we found the component
	  //now, iterate through all the respective files (functions)
	  //for the component.
	  std::vector<std::string>::iterator v;
	  for(v = mit->second.begin(); v!= mit->second.end(); v++){
	    instance_predictions.clear();
	    std::fstream datafile;
	    datafile.open((*v).c_str(), std::ios::in); 
	    //check for a file error 
	    if(datafile.bad()){
	      std::cerr<<"Error trying to open file: "<<(*v)<<std::cerr<<std::endl;
	      exit(0);
	    }
	    std::vector<double> inputs;  //vector of inputs
	    //we need the component name and function name.  We should be able to get this from
	    //the filename, which is in the form of: <component_name::function_name.[n]>,
	    //where .[n] is the processor id suffix.
	    std::string fname = *v; //get the filename
	    std::string func_name = fname.substr(fname.find("::") + 2);
	    func_name = func_name.substr(0, func_name.find("("));
	    std::string component_name = (*it)->getName();
	    std::cout<<"func_name: "<<func_name<<"\t"<<"comp_name: "<<component_name<<std::endl;
	    while(!datafile.eof()){
	      std::string line;
	      getline(datafile, line);
	      //make sure the line is not empty or a comment
	      if(line[0] != '#' && line[0] != '\0'){
		//remove the first two columns, which contain
		//MPI_Time and Compute_Time, respectively.
		//if there is a preceding tab, delete it
		if(line[0] == '\t')
		  line = line.substr(1);
		//take out the first two columns
		line = line.substr(line.find("\t") + 1);
		line = line.substr(line.find("\t") + 1);
		//now, any other columns are data we need to 
		//store in a double array
		double d;                //input
		//Now, go through and insert each parameter into 
		//the vector
		while(line[0] != '\0'){
		  if(line.find("\t") == std::string::npos){
		    //only one column - convert the entire string 
		    //to double
		    d = atof(line.c_str());
		    line = "\0";
		  }
		  else{
		    //more than one column
		    std::string input = line.substr(0, line.find("\t"));
		    d = atof(input.c_str());
		    //now, update the line  
		    line = line.substr(line.find("\t") + 1);
		  }
		  inputs.push_back(d);
		}//while - finished an input line
		//now we have a vector of input values for a single call to the 
		//current function.  Submit these inputs to the relevant 
		//performance model and get the performance prediciton
		std::map<std::string, double > temp;
		//find the correct component instance so we can get the model
		std::vector<modeling::AbstractFamilyMember *>::iterator vit;
		for(vit = cset.begin(); vit != cset.end(); vit++){
		  if((*vit)->getName() == component_name){
		    (*vit)->getPerfPrediction(inputs, func_name, temp);
		  }
		}//for
		//now, add those predictions to the current instance predictions
		std::map<std::string, double>::iterator iter;
		for(iter = temp.begin(); iter != temp.end(); iter++){
		  instance_predictions[iter->first] += iter->second;
		}//for
		//reset the inputs vector
		inputs.clear();
	      }//if - finished processing a line
	    }//done with a file
	    datafile.close();
	    std::cout<<"Predictions for "<<component_name<<"::"<<func_name<<": "<<instance_predictions["ExecutionTime"]<<std::endl;
	    //now, update the cumulative predictions from all instances in subset
	    std::map<std::string, double>::iterator it1;
	    for(it1 = instance_predictions.begin(); it1 != instance_predictions.end(); it1++){
	      predictions[it1->first] += it1->second;
	    }//for
	  }
	}//for - iterate through all functions of a given instance
	else
	  std::cout<<"Error.  No "<<(*it)->getName()
		   <<" entry found in the database."<<std::endl;
      }//for - done adding up all the predictions for an entire instance subset
      //compare this prediction with the current best set.  If it is better, update it.
      std::cout<<"---------------------------------------------------"<<std::endl;
      if(opt_result.size() == 0 || opt_result["ExecutionTime"] > predictions["ExecutionTime"]){
	opt_result = predictions;
	best_set = cset;
      }//if
    }
    return 1;
  }

  int Evaluator::evaluateOptimalComponentSet(std::vector<AbstractFamilyMember *> &best_set, std::map<std::string, double> &opt_result)
  {
    //We need to go through each component set (one instance from each family),
    //and compute the expected execution time.  Once these are done, return the 
    //component set with the minimum time.
    
    std::map<std::string, std::vector<AbstractFamilyMember *> >::iterator family_iter;
    std::vector<AbstractFamilyMember *>::iterator member_iter;
    family_iter=families.begin();
    for(member_iter = family_iter->second.begin(); member_iter != family_iter->second.end(); member_iter++){
      std::vector<AbstractFamilyMember *> set;
      set.push_back(*member_iter);
      recSelectMember(best_set, opt_result, set, ++family_iter);
      --family_iter;
    }
    return 1;
  }
}

