#include <map>
#include <string>
#include <vector>
#include "MinTreeNode.h"

namespace modeling{
  typedef int (*PerfModelPointer)(std::vector<double> array_dims, 
				  std::map<std::string,double> &outputs);

  //FamilyMember class.  A FamilyMember is an abstract class that represents a component 
  //instance.  A family consists of component instances (family members) that provide 
  //the same functionality via a different algorithm or data structure (i.e. the 
  //components can be interchanged for one another)
  class AbstractFamilyMember{
  public:
    AbstractFamilyMember(std::string n):name(n){}
    virtual ~AbstractFamilyMember(){}
    std::string getName(){ return name;}
    //Get a performance prediction based on a vector of input values for a given method.
    //The results are returned in a map, to allow for multiple metrics, such as 
    //wall clock time, cache hit rate, etc.
    virtual int getPerfPrediction(std::vector<double> dims, std::string method_name, 
				  std::map<std::string,double> &output)=0;
  private:
    std::string name;
  };

  //Evaluator class - allows a user to instantiate an evaluator in order to access
  //the library's functionality.
  class Evaluator{
  public:
    Evaluator();
    ~Evaluator();
    //read in database file and put entries in map database.  The component::routine 
    //is the key, and the filename of where the data is located is the keyed data.  
    //1 is returned on success, and -1 for a file I/0 error
    int setDBFile(std::string filename);
    
    //print to standard out a list of all component::method entries in the database.   
    void printAllDBEntries();
    
    //Add the given family to the map of families
    //A family is made up of possibly multiple family members.  
    //An instance is only allowed to belong to one family.  If a duplicate instance
    //name is detected (either being added to the same family, or another, it is ignored.
    int setFamily(std::string family_name, std::vector<AbstractFamilyMember *>f_members);
    
    //print to standard out a list of all families and their family members.
    void printFamilies();
    
    //return the size of the families map (ie the number of families -- generally the 
    //number of families should be the number of components being modeled, as there 
    //needs to be one instance per family)
    int getNumFamilies();
    
    //Take in the filename of the file containing an instance of a minimal component 
    //call-graph.  Recreate the tree using this file.  Return 1 for success, and -1 
    //for a file I/O error.
    int setMinimalNetwork(std::string filename);

    //Transform the minimalNetwork instance into a generalized tree using only family 
    //names.  Any families that are not used in the minimalFamilyCoreTree will be 
    //removed as unnecessary (insignificant) families.  The resulting families map will 
    //only contain the core families.  Return -1 if minimalNetworkTree is uninitialized.
    //return 1 on success.
    int createFamilyCoreTree();
    
    //print to standard out the core families and their family members.
    int printMinFamilies();

    //determine the optimal set of components to sovle the given problem.  Will 
    //iterate through each family member for a given family that is included in the 
    //minimal network.  Will store the best set of comonent instances (one for each 
    //family) in best_set, and the overall metric in opt_result
    int evaluateOptimalComponentSet(std::vector<AbstractFamilyMember *> &best_set, 
				    std::map<std::string, double> &opt_result);

  private:
    //A recursive helper routine used to recursively select a subset of instances in 
    //order to find the optimal subset.
    int recSelectMember(std::vector<AbstractFamilyMember *> &best_set, 
			std::map<std::string, double> & opt_result, 
			std::vector<AbstractFamilyMember *> &cset, 
			std::map<std::string, 
			std::vector<AbstractFamilyMember *> >::iterator family_iter);

    //A recursive helper routine used in creating the Family Core Tree.
    int recTransformChildren(MinTreeNode * tree);

    //a map of family names and vectors of AbstractFamilyMembers 
    std::map<std::string, std::vector<AbstractFamilyMember *> > families;

    //database map contains the component name (key) and a vector containing the file 
    //name(s) of where the proxy performance data is stored.  There may be multiple 
    //entries in the vector, as a component may have multiple functions being montiored.  
    std::map<std::string,std::vector<std::string> > database;

    //The representation of the  minimal tree. 
    MinTreeNode * mintree;

    //A vector that will contain the names of the families that make up the core 
    //components
    std::vector<std::string> min_families;
  };
}
